/* FichothequeLib_Tools  - Copyright (c) 2007-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.tools.format.parsers;

import net.fichotheque.addenda.Document;
import net.fichotheque.format.FormatContext;
import net.fichotheque.format.FormatSource;
import net.fichotheque.format.formatters.DocumentFormatter;
import net.fichotheque.tools.format.JsonParameters;
import net.fichotheque.tools.format.LangParameters;
import net.fichotheque.tools.format.catalogs.DocumentFormatterCatalog;
import net.fichotheque.tools.format.DefaultPatternCatalog;
import net.fichotheque.utils.FormatterUtils;
import net.mapeadores.util.format.FormatConstants;
import net.mapeadores.util.instruction.Argument;
import net.mapeadores.util.instruction.Instruction;
import net.mapeadores.util.instruction.InstructionResolver;
import net.mapeadores.util.instruction.InstructionResolverProvider;
import net.mapeadores.util.logging.ErrorMessageException;
import net.mapeadores.util.logging.MessageHandler;


/**
 *
 * @author Vincent Calame
 */
public final class DocumentFormatterParser {

    private final static JsonParameters DEFAULT_JSONPARAMETERS = JsonParameters.build("type", "id", "basename", "versions");

    private DocumentFormatterParser() {
    }

    public static DocumentFormatter parse(String pattern, FormatContext formatContext, boolean withWeightFilter, MessageHandler messageHandler) {
        if (pattern == null) {
            pattern = DefaultPatternCatalog.DOCUMENT;
        }
        try {
            return parse(pattern, formatContext, withWeightFilter);
        } catch (ErrorMessageException fe) {
            messageHandler.addMessage(FormatConstants.SEVERE_PATTERN, fe.getErrorMessage());
            return null;
        }
    }

    public static DocumentFormatter parse(String pattern, FormatContext formatContext, boolean withWeightFilter) throws ErrorMessageException {
        Object[] partArray = FormatterUtils.parsePattern(new InternalInstructionResolver(formatContext, withWeightFilter), pattern);
        return new InternalDocumentFormatter(partArray);
    }


    private static class InternalDocumentFormatter implements DocumentFormatter {

        private final Object[] partArray;

        private InternalDocumentFormatter(Object[] partArray) {
            this.partArray = partArray;
        }

        @Override
        public String formatDocument(Document document, int weight, FormatSource formatSource) {
            StringBuilder buf = new StringBuilder();
            int length = partArray.length;
            for (int i = 0; i < length; i++) {
                Object obj = partArray[i];
                if (obj instanceof String) {
                    buf.append((String) obj);
                    continue;
                }
                DocumentFormatter formatter = (DocumentFormatter) obj;
                buf.append(formatter.formatDocument(document, weight, formatSource));
            }
            return buf.toString();
        }

    }


    private static class InternalInstructionResolver implements InstructionResolver {

        private final InstructionResolverProvider provider;
        private final boolean withWeightFilter;

        private InternalInstructionResolver(FormatContext formatContext, boolean withWeightFilter) {
            this.provider = formatContext.getInstructionResolverProvider();
            this.withWeightFilter = withWeightFilter;
        }

        @Override
        public Object resolve(Instruction instruction) throws ErrorMessageException {
            InstructionResolver resolver = provider.getInstructionResolver(DocumentFormatter.class);
            if (resolver != null) {
                Object formatter = resolver.resolve(instruction);
                if (formatter != null) {
                    return formatter;
                }
            }
            Argument arg1 = instruction.get(0);
            String key = arg1.getKey();
            switch (key) {
                case "addenda":
                    return DocumentFormatterCatalog.ADDENDA;
                case "basename":
                    return DocumentFormatterCatalog.BASENAME;
                case "extensions":
                    return DocumentFormatterCatalog.EXTENSIONS;
                case "formsyntax":
                    if (withWeightFilter) {
                        return DocumentFormatterCatalog.WITHWEIGHTFILTER_FORMSYNTAX;
                    } else {
                        return DocumentFormatterCatalog.WITHOUTWEIGHTFILTER_FORMSYNTAX;
                    }
                case "id":
                    return DocumentFormatterCatalog.ID;
                case "json": {
                    JsonParameters jsonParameters = JsonParameters.fromInstruction(instruction, DEFAULT_JSONPARAMETERS);
                    LangParameters langParameters = LangParameters.fromInstruction(instruction, "langs");
                    return new DocumentFormatterCatalog.Json(jsonParameters, langParameters.getCustomLangContext());
                }
                case "weight":
                case "poids":
                    return DocumentFormatterCatalog.WEIGHT;
                default:
                    return null;

            }
        }

    }

}
