/* global Fapi, Fql */

/**
 * Contient des méthodes statiques pour l'interrogation de l'API du serveur BDF.
 * Les options sont les suivantes :
 * - requestParameters : paramètres supplémentaires passés à la requête
 * - callback: fonction appelée à la fin du traitement
 * 
 * @type Object
 */
Fapi.Ajax = {};


/**
 * Retourne l'identité du l'utilisateur actuellement connecté
 * 
 * @param {Fapi} fapi Configuration à utiliser
 * @param {Object} options Options de la requête
 * @param {function} options.callback Fonction de rappel
 * @param {Object} options.requestParameters Surcharge des paramètres de la requête
 * @param {Object} options.ignoreLogError Ne pas tenir compte des erreurs
 * @returns {undefined}
 */
Fapi.Ajax.ping = function (fapi, options) {
    var requestParameters = fapi.buildRequestParameters({jsonName: "ping", version: "action"}, options);
    $.ajax({
        url: fapi.getActionUrl("session"),
        method: "GET",
        dataType: "json",
        data: requestParameters,
        xhrFields: {
           withCredentials: true
        },
        success: function (data, textStatus) {
            if (options) {
                fapi.handleResponseData(data, false, options.callback);
            }
        }
    });
};


/**
 * Déconnecte l'utilisateur en cours
 * 
 * @param {Fapi} fapi Configuration à utiliser
 * @param {Object} options Options de la requête
 * @param {function} options.callback Fonction de rappel
 * @param {Object} options.requestParameters Surcharge des paramètres de la requête
 * @param {Object} options.ignoreLogError Ne pas tenir compte des erreurs
 * @returns {undefined}
 */
Fapi.Ajax.exit = function (fapi, options) {
    var requestParameters = fapi.buildRequestParameters({jsonName: "ping", version: "action"}, options);
    requestParameters["bdf-exit"] = 1;
    $.ajax({
        url: fapi.getActionUrl("session"),
        method: "GET",
        dataType: "json",
        data: requestParameters,
        xhrFields: {
           withCredentials: true
        },
        success: function (data, textStatus) {
            if (options) {
                fapi.handleResponseData(data, false, options.callback);
            }
        }
    });
};


/**
 * Récupère une fiches seule
 * 
 * @param {Fapi} fapi Configuration à utiliser
 * @param {string} corpus Nom du corpus
 * @param {number} id Numéro de la fiche
 * @param {function} options.callback Fonction de rappel
 * @param {Object} options.requestParameters Surcharge des paramètres de la requête
 * @param {Object} options.ignoreLogError Ne pas tenir compte des erreurs
 * @returns {undefined}
 */
Fapi.Ajax.loadFiche = function (fapi, options) {
    var endPoint = fapi.getEndPoint("fiche");
    var requestParameters = fapi.buildRequestParameters(endPoint, options);
    $.ajax({
        url: endPoint.url,
        method: "GET",
        dataType: "json",
        data: requestParameters,
        xhrFields: {
           withCredentials: fapi.isWithCredentials()
        },
        success: function (data, textStatus) {
            if (options) {
                fapi.handleResponseData(data, !(options.ignoreLogError), options.callback);
            }
        }
    });
};


/**
 * Récupère le tableau des fiches correspondantes aux critères de sélection
 * 
 * @param {Fapi} fapi Configuration à utiliser
 * @param {Fql.FicheQuery} ficheQuery 
 * @param {Object} options Options de la requête
 * @param {function} options.callback Fonction de rappel
 * @param {Object} options.requestParameters Surcharge des paramètres de la requête
 * @param {Object} options.ignoreLogError Ne pas tenir compte des erreurs
 * @returns {undefined}
 */
Fapi.Ajax.loadFicheArray = function (fapi, ficheQuery, options) {
    var endPoint = fapi.getEndPoint("fiche-array");
    var requestParameters = fapi.buildRequestParameters(endPoint, options);
    requestParameters.xml = ficheQuery.toXml(-999);
    $.ajax({
        url: endPoint.url,
        method: "GET",
        dataType: "json",
        data: requestParameters,
        xhrFields: {
           withCredentials: fapi.isWithCredentials()
        },
        success: function (data, textStatus) {
            if (options) {
                fapi.handleResponseData(data, !(options.ignoreLogError), options.callback);
            }
        }
    });
};


Fapi.Ajax.loadMotcle = function (fapi, options) {
    var endPoint = fapi.getEndPoint("motcle");
    var requestParameters = fapi.buildRequestParameters(endPoint, options);
    $.ajax({
        url: endPoint.url,
        method: "GET",
        dataType: "json",
        data: requestParameters,
        xhrFields: {
           withCredentials: fapi.isWithCredentials()
        },
        success: function (data, textStatus) {
            if (options) {
                fapi.handleResponseData(data, !(options.ignoreLogError), options.callback);
            }
        }
    });
};


/**
 * 
 * @param {Fapi} fapi Configuration à utiliser
 * @param {Fql.MotcleQuery} motcleQuery
 * @param {Object} options Options de la requête
 * @returns {undefined}
 */
Fapi.Ajax.loadMotcleArray = function (fapi, motcleQuery, options) {
    var endPoint = fapi.getEndPoint("motcle-array");
    var requestParameters = fapi.buildRequestParameters(endPoint, options);
    requestParameters.xml = motcleQuery.toXml(-999);
    $.ajax({
        url: endPoint.url,
        method: "GET",
        dataType: "json",
        data: requestParameters,
        xhrFields: {
           withCredentials: fapi.isWithCredentials()
        },
        success: function (data, textStatus) {
            if (options) {
                fapi.handleResponseData(data, !(options.ignoreLogError), options.callback);
            }
        }
    });
};


/**
 * 
 * @param {Fapi} fapi Configuration à utiliser
 * @param {String} thesaurusName Nom du thésaurus
 * @param {Object} options Options de la requête
 * @param {function} options.callback Fonction de rappel
 * @param {Object} options.requestParameters Surcharge des paramètres de la requête
 * @param {Object} options.ignoreLogError Ne pas tenir compte des erreurs
 * @returns {undefined}
 */
Fapi.Ajax.loadThesaurus = function (fapi, thesaurusName, options) {
    var endPoint = fapi.getEndPoint("thesaurus");
    var requestParameters = fapi.buildRequestParameters(endPoint, options);
    requestParameters.thesaurus = thesaurusName;        
    $.ajax({
        url: endPoint.url,
        method: "GET",
        dataType: "json",
        data: requestParameters,
        xhrFields: {
           withCredentials: fapi.isWithCredentials()
        },
        success: function (data, textStatus) {
            if (options) {
                fapi.handleResponseData(data, !(options.ignoreLogError), options.callback);
            }
        }
    });
};


Fapi.Ajax.loadFiches = function (fapi, queries, options) {
    var endPoint = fapi.getEndPoint("fiches");
    var requestParameters = fapi.buildRequestParameters(endPoint, options);
    var xml;
    if (typeof queries === "string") {
        xml = queries;
    } else if (queries instanceof Fql.FicheQuery) {
        xml = queries.toXml(-999);
    } else if (Array.isArray(queries)) {
        xml = "<queries>";
        for(let query of queries) {
            if (typeof query === "string") {
                xml += query;
            } else if (query instanceof Fql.FicheQuery) {
                xml  += query.toXml(-999);
            } else if (typeof query === "object") {
                let ficheQuery = new Fql.FicheQuery(query);
                xml  += ficheQuery.toXml(-999);
            }
        }
        xml += "</queries>";
    } else if (typeof queries === "object") {
        let ficheQuery = new Fql.FicheQuery(queries);
        xml = ficheQuery.toXml(-999);
    }
    requestParameters.xml = xml;
    $.ajax({
        url: endPoint.url,
        method: "GET",
        dataType: "json",
        data: requestParameters,
        xhrFields: {
           withCredentials: fapi.isWithCredentials()
        },
        success: function (data, textStatus) {
            if (options) {
                fapi.handleResponseData(data, !(options.ignoreLogError), options.callback);
            }
        }
    });
};
