/* UtilLib - Copyright (c) 2016-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.mapeadores.util.logging;

import java.util.ArrayList;
import java.util.List;
import net.mapeadores.util.localisation.Message;


/**
 *
 * @author Vincent Calame
 */
public class SourceLogBuilder implements MessageHandler {

    private final String name;
    private final List<InternalSourceMessage> sourceMessageList = new ArrayList<InternalSourceMessage>();

    public SourceLogBuilder() {
        this.name = "";
    }

    public SourceLogBuilder(String name) {
        this.name = name;
    }

    public boolean hasMessage() {
        return (!sourceMessageList.isEmpty());
    }

    public boolean isEmpty() {
        return (sourceMessageList.isEmpty());
    }

    @Override
    public void addMessage(String category, Message message) {
        if (category == null) {
            throw new IllegalArgumentException("category is null");
        }
        if (message == null) {
            throw new IllegalArgumentException("message is null");
        }
        sourceMessageList.add(new InternalSourceMessage(category, message));
    }

    public List<SourceMessage> filter(String category) {
        List<SourceMessage> list = new ArrayList<SourceMessage>();
        for (InternalSourceMessage sourceMessage : sourceMessageList) {
            if (sourceMessage.getCategory().startsWith(category)) {
                list.add(sourceMessage);
            }
        }
        return list;
    }

    public List<SourceMessage> filterWarning() {
        return filter("warning.");
    }

    public List<SourceMessage> filterSevere() {
        return filter("severe.");
    }

    public String toRawString() {
        StringBuilder buf = new StringBuilder();
        for (SourceMessage sourceMessage : sourceMessageList) {
            if (buf.length() > 0) {
                buf.append("\n");
            }
            buf.append(sourceMessage.getMessageKey());
            Object[] values = sourceMessage.getMessageValues();
            int length = values.length;
            if (length > 0) {
                for (int i = 0; i < length; i++) {
                    if (i > 0) {
                        buf.append(" / ");
                    } else {
                        buf.append(": ");
                    }
                    buf.append(values[i].toString());
                }
            }
        }
        return buf.toString();
    }

    public SourceLog toSourceLog() {
        List<SourceMessage> finalList = LogUtils.wrap(sourceMessageList.toArray(new SourceMessage[sourceMessageList.size()]));
        return new InternalSourceLog(name, finalList);
    }

    public static SourceLogBuilder init() {
        return new SourceLogBuilder();
    }

    public static SourceLogBuilder init(String name) {
        return new SourceLogBuilder(name);
    }


    private static class InternalSourceLog implements SourceLog {

        private final String name;
        private final List<SourceMessage> sourceMessageList;

        private InternalSourceLog(String name, List<SourceMessage> sourceMessageList) {
            this.name = name;
            this.sourceMessageList = sourceMessageList;
        }

        @Override
        public String getName() {
            return name;
        }

        @Override
        public List<SourceMessage> getSourceMessageList() {
            return sourceMessageList;
        }

    }


    private static class InternalSourceMessage implements SourceMessage {

        private final String category;
        private final Message message;

        private InternalSourceMessage(String category, Message message) {
            this.category = category;
            this.message = message;
        }

        @Override
        public String getCategory() {
            return category;
        }

        @Override
        public String getMessageKey() {
            return message.getMessageKey();
        }

        @Override
        public Object[] getMessageValues() {
            return message.getMessageValues();
        }

    }

}
