/* FichothequeLib_Tools - Copyright (c) 2013-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.tools.format.parsers;

import net.fichotheque.SubsetKey;
import net.fichotheque.corpus.fiche.FicheBlocks;
import net.fichotheque.format.FormatContext;
import net.fichotheque.format.FormatSource;
import net.fichotheque.format.formatters.ExtractionFormatter;
import net.fichotheque.format.formatters.FicheBlockFormatter;
import net.fichotheque.tools.format.DefaultPatternCatalog;
import net.fichotheque.tools.format.ValuesParameters;
import net.fichotheque.tools.format.catalogs.FicheBlockFormatterCatalog;
import net.fichotheque.utils.FormatterUtils;
import net.mapeadores.util.format.FormatConstants;
import net.mapeadores.util.instruction.Argument;
import net.mapeadores.util.instruction.Instruction;
import net.mapeadores.util.instruction.InstructionResolver;
import net.mapeadores.util.logging.ErrorMessageException;
import net.mapeadores.util.logging.MessageHandler;


/**
 *
 * @author Vincent Calame
 */
public final class FicheBlockFormatterParser {

    private FicheBlockFormatterParser() {
    }

    public static FicheBlockFormatter parse(String pattern, FormatContext formatContext, MessageHandler messageHandler) {
        if (pattern == null) {
            pattern = DefaultPatternCatalog.FICHEBLOCK;
        }
        try {
            return parse(pattern, formatContext);
        } catch (ErrorMessageException fe) {
            messageHandler.addMessage(FormatConstants.SEVERE_PATTERN, fe.getErrorMessage());
            return null;
        }
    }


    public static FicheBlockFormatter parse(String pattern, FormatContext formatContext) throws ErrorMessageException {
        Object[] partArray = FormatterUtils.parsePattern(new InternalInstructionResolver(formatContext), pattern);
        return new InternalFicheBlockFormatter(partArray);
    }


    private static class InternalFicheBlockFormatter implements FicheBlockFormatter {

        private final Object[] partArray;

        private InternalFicheBlockFormatter(Object[] partArray) {
            this.partArray = partArray;
        }

        @Override
        public String formatFicheBlocks(FicheBlocks ficheBlocks, FormatSource formatSource, SubsetKey defaultCorpusKey) {
            StringBuilder buf = new StringBuilder();
            int length = partArray.length;
            for (int i = 0; i < length; i++) {
                Object obj = partArray[i];
                if (obj instanceof String) {
                    buf.append((String) obj);
                    continue;
                }
                FicheBlockFormatter formatter = (FicheBlockFormatter) obj;
                buf.append(formatter.formatFicheBlocks(ficheBlocks, formatSource, defaultCorpusKey));
            }
            return buf.toString();
        }

    }


    private static class InternalInstructionResolver implements InstructionResolver {

        private final FormatContext formatContext;

        private InternalInstructionResolver(FormatContext formatContext) {
            this.formatContext = formatContext;
        }

        @Override
        public Object resolve(Instruction instruction) throws ErrorMessageException {
            InstructionResolver resolver = formatContext.getInstructionResolverProvider().getInstructionResolver(ExtractionFormatter.class);
            if (resolver != null) {
                Object formatter = resolver.resolve(instruction);
                if (formatter != null) {
                    return formatter;
                }
            }
            Argument arg1 = instruction.get(0);
            String key = arg1.getKey();
            switch (key) {
                case "formsyntax":
                    return FicheBlockFormatterCatalog.FORMSYNTAX;
                case "text":
                    return FicheBlockFormatterCatalog.TEXT;
                case "values":
                    return new FicheBlockFormatterCatalog.Values(ValuesParameters.fromInstruction(instruction));
                case "xml":
                    return formatContext.getFicheBlockFormatter(FormatContext.XML_SPECIAL_FORMATTER_NAME, instruction.toOptionMap(1));
                case "transformation":
                case "xslt":
                    String name = arg1.getValue();
                    FicheBlockFormatter ficheBlockFormatter = formatContext.getFicheBlockFormatter(name, instruction.toOptionMap(1));
                    if (ficheBlockFormatter == null) {
                        throw new ErrorMessageException("_ error.unknown.template", name);
                    }
                    return ficheBlockFormatter;
                default:
                    return null;
            }
        }

    }


}
