/* FichothequeLib_Tools - Copyright (c) 2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.tools.format.catalogs;

import java.io.IOException;
import net.fichotheque.album.Illustration;
import net.fichotheque.format.FormatSource;
import net.fichotheque.format.formatters.IllustrationFormatter;
import net.fichotheque.json.IllustrationJson;
import net.fichotheque.tools.format.JsonParameters;
import net.mapeadores.util.exceptions.ShouldNotOccurException;
import net.mapeadores.util.json.JSONWriter;


/**
 *
 * @author Vincent Calame
 */
public final class IllustrationFormatterCatalog {

    public final static IllustrationFormatter ALBUM = (illustration, weight, formatSource) -> {
        return illustration.getSubsetName();
    };
    public final static IllustrationFormatter FORMAT = (illustration, weight, formatSource) -> {
        return illustration.getFormatType();
    };
    public final static IllustrationFormatter ID = (illustration, weight, formatSource) -> {
        return String.valueOf(illustration.getId());
    };
    public final static IllustrationFormatter WEIGHT = (illustration, weight, formatSource) -> {
        return String.valueOf(weight);
    };
    public final static IllustrationFormatter WITHWEIGHTFILTER_FORMSYNTAX = new InternalFormSyntax(true);
    public final static IllustrationFormatter WITHOUTWEIGHTFILTER_FORMSYNTAX = new InternalFormSyntax(false);


    private IllustrationFormatterCatalog() {

    }


    public static class Json implements IllustrationFormatter {

        private final JsonParameters jsonParameters;

        public Json(JsonParameters jsonParameters) {
            this.jsonParameters = jsonParameters;
        }

        @Override
        public String formatIllustration(Illustration illustration, int weight, FormatSource formatSource) {
            StringBuilder buf = new StringBuilder();
            JSONWriter jsonWriter = new JSONWriter(buf);
            try {
                jsonWriter.object();
                IllustrationJson.properties(jsonWriter, illustration, jsonParameters.getPropertyEligibility());
                jsonWriter.endObject();
            } catch (IOException ioe) {
                throw new ShouldNotOccurException(ioe);
            }
            return buf.toString();
        }

    }


    private static class InternalFormSyntax implements IllustrationFormatter {

        private final boolean withWeightFilter;

        private InternalFormSyntax(boolean withWeightFilter) {
            this.withWeightFilter = withWeightFilter;
        }

        @Override
        public String formatIllustration(Illustration illustration, int weight, FormatSource formatSource) {
            if ((withWeightFilter) || (weight == 1)) {
                return String.valueOf(illustration.getId());
            } else {
                StringBuilder buf = new StringBuilder();
                buf.append(String.valueOf(illustration.getId()));
                buf.append(" <");
                buf.append(weight);
                buf.append(">");
                return buf.toString();
            }
        }

    }

}
