/* FichothequeLib_Tools - Copyright (c) 2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.tools.format.catalogs;

import java.io.IOException;
import net.fichotheque.corpus.FicheMeta;
import net.fichotheque.exportation.table.Cell;
import net.fichotheque.format.FormatSource;
import net.fichotheque.format.formatters.FicheMetaFormatter;
import net.fichotheque.json.FicheMetaJson;
import net.fichotheque.tools.format.FormatterEngineUtils;
import net.fichotheque.tools.format.JsonParameters;
import net.mapeadores.util.exceptions.ShouldNotOccurException;
import net.mapeadores.util.json.JSONWriter;
import net.mapeadores.util.localisation.Lang;


/**
 *
 * @author Vincent Calame
 */
public final class FicheMetaFormatterCatalog {

    public final static FicheMetaFormatter CORPUS = (ficheMeta, weight, formatSource) -> {
        return ficheMeta.getSubsetName();
    };
    public final static FicheMetaFormatter ID = (ficheMeta, weight, formatSource) -> {
        return String.valueOf(ficheMeta.getId());
    };
    public final static FicheMetaFormatter LANG = (ficheMeta, weight, formatSource) -> {
        Lang lang = ficheMeta.getLang();
        if (lang != null) {
            return lang.toString();
        } else {
            return "";
        }
    };
    public final static FicheMetaFormatter WEIGHT = (ficheMeta, weight, formatSource) -> {
        return String.valueOf(weight);
    };
    public final static FicheMetaFormatter TITLE = (ficheMeta, weight, formatSource) -> {
        return ficheMeta.getTitle();
    };

    private FicheMetaFormatterCatalog() {

    }


    public static class Json implements FicheMetaFormatter {

        private final JsonParameters jsonParameters;

        public Json(JsonParameters jsonParameters) {
            this.jsonParameters = jsonParameters;
        }

        @Override
        public String formatFicheMeta(FicheMeta ficheMeta, int weight, FormatSource formatSource) {
            Cell[] cellArray = FormatterEngineUtils.resolve(ficheMeta, jsonParameters, formatSource);
            StringBuilder buf = new StringBuilder();
            JSONWriter jw = new JSONWriter(buf);
            try {
                jw.object();
                FicheMetaJson.properties(jw, ficheMeta, jsonParameters.getPropertyEligibility(), cellArray);
                jw.endObject();
            } catch (IOException ioe) {
                throw new ShouldNotOccurException(ioe);
            }
            return buf.toString();
        }

    }


    public static class FormSyntax implements FicheMetaFormatter {

        private final boolean withWeightFilter;
        private final boolean withCorpusName;

        public FormSyntax(boolean withWeightFilter, boolean withCorpusName) {
            this.withWeightFilter = withWeightFilter;
            this.withCorpusName = withCorpusName;
        }

        @Override
        public String formatFicheMeta(FicheMeta ficheMeta, int weight, FormatSource formatSource) {
            StringBuilder buf = new StringBuilder();
            if (withCorpusName) {
                buf.append(ficheMeta.getSubsetName());
                buf.append('/');
            }
            buf.append(String.valueOf(ficheMeta.getId()));
            if ((!withWeightFilter) && (weight > 1)) {
                buf.append(" <");
                buf.append(weight);
                buf.append(">");
            }
            return buf.toString();
        }

    }

}
