/* FichothequeLib_Tools - Copyright (c) 2011-2021 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.tools.conversion;

import java.util.List;
import net.fichotheque.corpus.fiche.AmountItem;
import net.fichotheque.corpus.fiche.Cdatadiv;
import net.fichotheque.corpus.fiche.Code;
import net.fichotheque.corpus.fiche.CountryItem;
import net.fichotheque.corpus.fiche.DateItem;
import net.fichotheque.corpus.fiche.Div;
import net.fichotheque.corpus.fiche.EmailItem;
import net.fichotheque.corpus.fiche.FicheBlock;
import net.fichotheque.corpus.fiche.FicheBlocks;
import net.fichotheque.corpus.fiche.FicheItem;
import net.fichotheque.corpus.fiche.FicheItems;
import net.fichotheque.corpus.fiche.GeopointItem;
import net.fichotheque.corpus.fiche.ImageItem;
import net.fichotheque.corpus.fiche.Item;
import net.fichotheque.corpus.fiche.LanguageItem;
import net.fichotheque.corpus.fiche.Li;
import net.fichotheque.corpus.fiche.LinkItem;
import net.fichotheque.corpus.fiche.Ln;
import net.fichotheque.corpus.fiche.NumberItem;
import net.fichotheque.corpus.fiche.P;
import net.fichotheque.corpus.fiche.ParaItem;
import net.fichotheque.corpus.fiche.ParagraphBlock;
import net.fichotheque.corpus.fiche.PersonItem;
import net.fichotheque.corpus.fiche.S;
import net.fichotheque.corpus.fiche.Table;
import net.fichotheque.corpus.fiche.Td;
import net.fichotheque.corpus.fiche.Tr;
import net.fichotheque.corpus.fiche.Ul;
import net.fichotheque.utils.FicheUtils;
import net.mapeadores.util.models.EmailCore;


/**
 *
 * @author Vincent Calame
 */
public final class ConversionUtils {

    private ConversionUtils() {
    }

    public static FicheBlock toFicheBlock(FicheItem ficheItem) {
        P prg = new P();
        if (ficheItem instanceof Item) {
            prg.addText(((Item) ficheItem).getValue());
        } else if (ficheItem instanceof ParaItem) {
            for (Object obj : (ParaItem) ficheItem) {
                if (obj instanceof String) {
                    prg.addText((String) obj);
                } else if (obj instanceof S) {
                    prg.addS((S) obj);
                }
            }
        } else if (ficheItem instanceof CountryItem) {
            prg.addText(((CountryItem) ficheItem).getCountry().toString());
        } else if (ficheItem instanceof LanguageItem) {
            prg.addText(((LanguageItem) ficheItem).getLang().toString());
        } else if (ficheItem instanceof DateItem) {
            prg.addText(((DateItem) ficheItem).getDate().toISOString());
        } else if (ficheItem instanceof GeopointItem) {
            GeopointItem geopoint = (GeopointItem) ficheItem;
            prg.addText(geopoint.getLatitude().toString() + " " + geopoint.getLongitude().toString());
        } else if (ficheItem instanceof ImageItem) {
            ImageItem image = (ImageItem) ficheItem;
            S span = new S(S.IMAGE);
            span.setRef(image.getSrc());
            span.setValue(image.getAlt());
            prg.addS(span);
        } else if (ficheItem instanceof LinkItem) {
            LinkItem link = (LinkItem) ficheItem;
            S span = new S(S.LINK);
            span.setRef(link.getHref());
            span.setValue(link.getTitle());
            prg.addS(span);
        } else if (ficheItem instanceof EmailItem) {
            EmailCore emailCore = ((EmailItem) ficheItem).getEmailCore();
            S span = new S(S.LINK);
            span.setRef("mailto:" + emailCore.getAddrSpec());
            span.setValue(emailCore.getRealName());
            prg.addS(span);
        } else if (ficheItem instanceof AmountItem) {
            AmountItem amountItem = (AmountItem) ficheItem;
            prg.addText(amountItem.getCurrency().getCurrencyCode() + " " + amountItem.getDecimal().toString());
        } else if (ficheItem instanceof NumberItem) {
            prg.addText(((NumberItem) ficheItem).getDecimal().toString());
        } else if (ficheItem instanceof PersonItem) {
            PersonItem personItem = (PersonItem) ficheItem;
            String redacteurGlobalId = personItem.getRedacteurGlobalId();
            if (redacteurGlobalId != null) {
                prg.addText(redacteurGlobalId);
            } else {
                prg.addText(personItem.getPersonCore().toStandardStyle());
            }
        } else {
            prg.addText("Unknown : " + ficheItem.getClass().getName());
        }
        return prg;
    }

    public static boolean appendPara(List<ParaItem> list, FicheBlocks ficheBlocks) {
        boolean onlyP = true;
        int size = ficheBlocks.size();
        for (int i = 0; i < size; i++) {
            FicheBlock ficheBlock = ficheBlocks.get(i);
            if (!(ficheBlock instanceof P)) {
                onlyP = false;
            }
            if (ficheBlock instanceof ParagraphBlock) {
                ParaItem para = toPara((ParagraphBlock) ficheBlock);
                if (para != null) {
                    list.add(para);
                }
            } else if (ficheBlock instanceof Div) {
                appendPara(list, ((Div) ficheBlock));
            } else if (ficheBlock instanceof Cdatadiv) {
                String s = ((Cdatadiv) ficheBlock).getCdata();
                if (s.length() > 0) {
                    ParaItem.Builder builder = new ParaItem.Builder();
                    builder.addText(s);
                    list.add(builder.toParaItem());
                }
            } else if (ficheBlock instanceof Ul) {
                for (Li li : (Ul) ficheBlock) {
                    appendPara(list, li);
                }
            } else if (ficheBlock instanceof Code) {
                for (Ln ln : (Code) ficheBlock) {
                    ParaItem para = toPara(ln);
                    if (para != null) {
                        list.add(para);
                    }
                }
            } else if (ficheBlock instanceof Table) {
                for (Tr tr : (Table) ficheBlock) {
                    for (Td td : tr) {
                        ParaItem para = toPara(td);
                        if (para != null) {
                            list.add(para);
                        }
                    }
                }
            }
        }
        return onlyP;
    }

    public static ParaItem toPara(ParagraphBlock paragraphBlock) {
        ParaItem.Builder builder = new ParaItem.Builder();
        if (paragraphBlock.isEmpty()) {
            return null;
        }
        for (Object obj : paragraphBlock) {
            if (obj instanceof S) {
                builder.addS((S) obj);
            } else if (obj instanceof String) {
                builder.addText((String) obj);
            }
        }
        return builder.toParaItem();
    }

    public static FicheItems toFicheItems(List<ParaItem> list) {
        if (list.isEmpty()) {
            return FicheUtils.EMPTY_FICHEITEMS;
        }
        int size = list.size();
        if (size == 1) {
            return FicheUtils.toFicheItems(list.get(0));
        }
        FicheItem[] array = new FicheItem[size];
        int p = 0;
        for (ParaItem para : list) {
            array[p] = para;
            p++;
        }
        return FicheUtils.toFicheItems(array);
    }


}
