/* FichothequeLib_API - Copyright (c) 2007-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.corpus.metadata;

import java.text.ParseException;
import net.mapeadores.util.localisation.Lang;
import net.mapeadores.util.money.ExtendedCurrency;


/**
 *
 * @author Vincent Calame
 */
public class SubfieldKey {

    public static final String SURNAME_SUBTYPE = "surname";
    public static final String FORENAME_SUBTYPE = "forename";
    public static final String SURNAMEFIRST_SUBTYPE = "surnamefirst";
    public static final String NONLATIN_SUBTYPE = "nonlatin";
    public static final String VALUE_SUBTYPE = "value";
    public static final String CURRENCY_SUBTYPE = "currency";
    public static final String LATITUDE_SUBTYPE = "lat";
    public static final String LONGITUDE_SUBTYPE = "lon";
    public static final String SRC_SUBTYPE = "src";
    public static final String ALT_SUBTYPE = "alt";
    public static final String TITLE_SUBTYPE = "title";
    public static final String OTHERS_SUBTYPE = "others";
    public static final String AMOUNT_SUBTYPE = "amount";
    public static final String LANG_SUBTYPE = "lang";
    protected final FieldKey fieldKey;
    protected final String subtype;

    protected SubfieldKey(FieldKey fieldKey, String subtype) {
        this.fieldKey = fieldKey;
        this.subtype = subtype;
    }

    public FieldKey getFieldKey() {
        return fieldKey;
    }

    public String getSubtype() {
        return subtype;
    }

    @Override
    public int hashCode() {
        return fieldKey.hashCode() + subtype.hashCode();
    }

    @Override
    public boolean equals(Object other) {
        if (other == null) {
            return false;
        }
        if (other == this) {
            return true;
        }
        if (other.getClass() != this.getClass()) {
            return false;
        }
        SubfieldKey fk = (SubfieldKey) other;
        if (!fk.subtype.equals(this.subtype)) {
            return false;
        }
        return fk.fieldKey.equals(this.fieldKey);
    }

    public String getKeyString() {
        return toString();
    }

    @Override
    public String toString() {
        return fieldKey.getKeyString() + "_" + subtype;
    }

    public static SubfieldKey toSubfieldKey(FieldKey fieldKey, String type) {
        type = checkSubtype(type);
        switch (type) {
            case AMOUNT_SUBTYPE:
                throw new IllegalArgumentException("Not use with AMOUNT_SUBTYPE");
            case LANG_SUBTYPE:
                throw new IllegalArgumentException("Not use with LANG_SUBTYPE");
        }
        return new SubfieldKey(fieldKey, type);
    }

    public static SubfieldKey parse(String s) throws ParseException {
        int idx0 = s.indexOf('_');
        if (idx0 == -1) {
            throw new ParseException("No '_' character", 0);
        }
        int idx1 = s.indexOf('_', idx0 + 1);
        if (idx1 == -1) {
            throw new ParseException("Missing second '_' character", idx0);
        }
        FieldKey fieldKey = FieldKey.parse(s.substring(0, idx1));
        String subtype = s.substring(idx1 + 1);
        int idx2 = subtype.indexOf('_');
        String secondPart = "";
        if (idx2 > -1) {
            secondPart = subtype.substring(idx2 + 1);
            subtype = subtype.substring(0, idx2);
        }
        try {
            subtype = checkSubtype(subtype);
        } catch (IllegalArgumentException iae) {
            throw new ParseException("Wrong subtype = " + subtype, idx1 + 1);
        }
        switch (subtype) {
            case AMOUNT_SUBTYPE: {
                try {
                    ExtendedCurrency currency = ExtendedCurrency.parse(secondPart);
                    return new AmountSubfieldKey(fieldKey, currency);
                } catch (ParseException pe) {
                    throw new ParseException("Wrong currency = " + secondPart, idx2 + 1);
                }
            }
            case LANG_SUBTYPE: {
                try {
                    Lang lang = Lang.parse(secondPart);
                    return new LangSubfieldKey(fieldKey, lang);
                } catch (ParseException pe) {
                    throw new ParseException("Wrong lang = " + secondPart, idx2 + 1);
                }
            }
            default:
                return new SubfieldKey(fieldKey, subtype);
        }
    }

    /**
     * Équivalent de parse() mais en envoyant IllegalArgumentException plutôt
     * que ParseException. à utiliser quand on est sûr de la syntaxe et permet
     * d'éviter un try {} catch {}
     */
    public static SubfieldKey build(String s) {
        try {
            return parse(s);
        } catch (ParseException pe) {
            throw new IllegalArgumentException(pe.getMessage());
        }
    }

    public static String checkSubtype(String subtype) {
        switch (subtype) {
            case SURNAME_SUBTYPE:
            case "nom":
                return SURNAME_SUBTYPE;
            case FORENAME_SUBTYPE:
            case "prenom":
                return FORENAME_SUBTYPE;
            case SURNAMEFIRST_SUBTYPE:
            case "nomavant":
                return SURNAMEFIRST_SUBTYPE;
            case NONLATIN_SUBTYPE:
            case "original":
                return NONLATIN_SUBTYPE;
            case VALUE_SUBTYPE:
                return VALUE_SUBTYPE;
            case CURRENCY_SUBTYPE:
            case "cur":
                return CURRENCY_SUBTYPE;
            case LATITUDE_SUBTYPE:
            case "latitude":
                return LATITUDE_SUBTYPE;
            case LONGITUDE_SUBTYPE:
            case "longitude":
                return LONGITUDE_SUBTYPE;
            case SRC_SUBTYPE:
                return SRC_SUBTYPE;
            case ALT_SUBTYPE:
                return ALT_SUBTYPE;
            case TITLE_SUBTYPE:
                return TITLE_SUBTYPE;
            case OTHERS_SUBTYPE:
                return OTHERS_SUBTYPE;
            case AMOUNT_SUBTYPE:
            case "montant":
                return AMOUNT_SUBTYPE;
            case LANG_SUBTYPE:
                return LANG_SUBTYPE;
            default:
                throw new IllegalArgumentException("Wrong subtype = " + subtype);
        }

    }

    public static boolean isLegalSubfield(CorpusField corpusField, String subfieldType) {
        switch (corpusField.getCategory()) {
            case FieldKey.PROP_CATEGORY: {
                switch (corpusField.getFicheItemType()) {
                    case CorpusField.PERSON_FIELD:
                        return isLegalPersonProp(subfieldType);
                    case CorpusField.AMOUNT_FIELD:
                        return isLegalAmountProp(subfieldType);
                    case CorpusField.GEOPOINT_FIELD:
                        return isLegalGeopointProp(subfieldType);
                    case CorpusField.IMAGE_FIELD:
                        return isLegalImageProp(subfieldType);
                    default:
                        return false;
                }
            }
            case FieldKey.INFO_CATEGORY: {
                switch (corpusField.getFicheItemType()) {
                    case CorpusField.AMOUNT_FIELD:
                        return isLegalAmountInfo(subfieldType);
                    default:
                        return false;
                }
            }
            default:
                return false;
        }
    }


    private static boolean isLegalImageProp(String subfieldType) {
        switch (subfieldType) {
            case SRC_SUBTYPE:
            case ALT_SUBTYPE:
            case TITLE_SUBTYPE:
                return true;
            default:
                return false;
        }
    }

    private static boolean isLegalGeopointProp(String subfieldType) {
        switch (subfieldType) {
            case LATITUDE_SUBTYPE:
            case LONGITUDE_SUBTYPE:
                return true;
            default:
                return false;
        }
    }

    private static boolean isLegalPersonProp(String subfieldType) {
        switch (subfieldType) {
            case SURNAME_SUBTYPE:
            case FORENAME_SUBTYPE:
            case SURNAMEFIRST_SUBTYPE:
            case NONLATIN_SUBTYPE:
                return true;
            default:
                return false;
        }
    }

    private static boolean isLegalAmountProp(String subfieldType) {
        switch (subfieldType) {
            case VALUE_SUBTYPE:
            case CURRENCY_SUBTYPE:
                return true;
            default:
                return false;
        }

    }

    private static boolean isLegalAmountInfo(String subfieldType) {
        switch (subfieldType) {
            case AMOUNT_SUBTYPE:
            case OTHERS_SUBTYPE:
                return true;
            default:
                return false;
        }

    }

}
