/* BdfServer - Copyright (c) 2009-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package fr.exemole.bdfserver.tools.ficheform.builders.field;

import fr.exemole.bdfserver.api.ficheform.FicheFormParameters;
import fr.exemole.bdfserver.api.ficheform.field.HiddenFieldEntry;
import fr.exemole.bdfserver.api.ui.FieldUi;
import fr.exemole.bdfserver.tools.L10nUtils;
import fr.exemole.bdfserver.tools.ficheform.FicheFormUtils;
import net.fichotheque.corpus.metadata.CorpusField;
import net.fichotheque.pointeurs.FichePointeur;
import net.mapeadores.util.attr.AttributeUtils;
import net.mapeadores.util.attr.Attributes;


/**
 *
 * @author Vincent Calame
 */
public class HiddenFieldEntryBuilder {

    private final CorpusField corpusField;
    private String label = "";
    private boolean mandatory = false;
    private String value = "";
    private Attributes attributes = AttributeUtils.EMPTY_ATTRIBUTES;

    public HiddenFieldEntryBuilder(CorpusField corpusField) {
        this.corpusField = corpusField;
    }

    public HiddenFieldEntryBuilder setLabel(String label) {
        if (label == null) {
            throw new IllegalArgumentException("label is null");
        }
        this.label = label;
        return this;
    }

    public HiddenFieldEntryBuilder setMandatory(boolean mandatory) {
        this.mandatory = mandatory;
        return this;
    }

    public HiddenFieldEntryBuilder setAttributes(Attributes attributes) {
        if (attributes == null) {
            this.attributes = AttributeUtils.EMPTY_ATTRIBUTES;
        } else {
            this.attributes = attributes;
        }
        return this;
    }

    public HiddenFieldEntryBuilder setValue(String value) {
        if (value == null) {
            throw new IllegalArgumentException("value is null");
        }
        this.value = value;
        return this;
    }

    public HiddenFieldEntry toHiddenFieldEntry() {
        return new InternalHiddenFieldEntry(corpusField, label, mandatory, attributes, value);
    }

    public static HiddenFieldEntryBuilder check(FichePointeur fichePointeur, FicheFormParameters ficheFormParameters, CorpusField corpusField, FieldUi fieldUi, String defaultValue) {
        String value;
        if (fichePointeur.isEmpty()) {
            if (defaultValue != null) {
                value = defaultValue;
            } else {
                value = FicheFormUtils.getDefVal(fichePointeur, ficheFormParameters, fieldUi);
            }
        } else {
            value = FicheFormUtils.getStringValue(fichePointeur, corpusField, ficheFormParameters);
        }
        if ((value.isEmpty()) && (fieldUi.isObsolete())) {
            return null;
        }
        return init(corpusField)
                .setValue(value)
                .setLabel(L10nUtils.toLabelString(ficheFormParameters, corpusField))
                .setAttributes(fieldUi.getAttributes());
    }

    public static HiddenFieldEntryBuilder init(CorpusField corpusField) {
        return new HiddenFieldEntryBuilder(corpusField);
    }


    private static class InternalHiddenFieldEntry implements HiddenFieldEntry {

        private final CorpusField corpusField;
        private final String label;
        private final boolean mandatory;
        private final Attributes attributes;
        private final String value;

        private InternalHiddenFieldEntry(CorpusField corpusField, String label, boolean mandatory, Attributes attributes, String value) {
            this.corpusField = corpusField;
            this.label = label;
            this.mandatory = mandatory;
            this.attributes = attributes;
            this.value = value;
        }

        @Override
        public CorpusField getCorpusField() {
            return corpusField;
        }

        @Override
        public String getLabel() {
            return label;
        }

        @Override
        public boolean isMandatory() {
            return mandatory;
        }

        @Override
        public Attributes getAttributes() {
            return attributes;
        }

        @Override
        public String getValue() {
            return value;
        }

    }

}
