/* FichothequeLib_Xml - Copyright (c) 2009-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.xml.storage;

import java.io.IOException;
import java.util.List;
import net.fichotheque.corpus.fiche.FicheAPI;
import net.fichotheque.corpus.fiche.FicheItem;
import net.fichotheque.corpus.fiche.FicheItems;
import net.fichotheque.corpus.fiche.Info;
import net.fichotheque.corpus.fiche.ParaItem;
import net.fichotheque.corpus.fiche.Prop;
import net.fichotheque.corpus.fiche.Section;
import net.mapeadores.util.xml.XMLPart;
import net.mapeadores.util.xml.XMLUtils;
import net.mapeadores.util.xml.XMLWriter;


/**
 *
 * @author Vincent Calame
 */
public class FicheStorageXMLPart extends XMLPart {

    private final FicheItemStorageXMLPart ficheItemXMLPart;
    private final FicheBlockStorageXMLPart ficheBlockStorageXMLPart;

    public FicheStorageXMLPart(XMLWriter xmlWriter) {
        super(xmlWriter);
        ficheItemXMLPart = new FicheItemStorageXMLPart(this);
        ficheBlockStorageXMLPart = new FicheBlockStorageXMLPart(this);
    }

    public void appendFiche(FicheAPI fiche) throws IOException {
        startOpenTag("fiche");
        XMLUtils.addXmlLangAttribute(this, fiche.getLang());
        endOpenTag();
        addEntete(fiche);
        addCorpsdefiche(fiche);
        closeTag("fiche");
    }

    private void addEntete(FicheAPI fiche) throws IOException {
        openTag("entete");
        addSimpleElement("titre", fiche.getTitle());
        ParaItem subtitlePara = fiche.getSubtitle();
        if (subtitlePara != null) {
            startOpenTag("soustitre");
            endOpenTag();
            ficheBlockStorageXMLPart.addTextContent(subtitlePara);
            closeTag("soustitre", false);
        }
        for (Prop prop : fiche.getPropList()) {
            addProp(prop);
        }
        for (Info info : fiche.getInfoList()) {
            addInfo(info);
        }
        FicheItems redacteursItemList = fiche.getOwners();
        if (redacteursItemList != null) {
            openTag("redacteurs");
            ficheItemXMLPart.addFicheItems(redacteursItemList);
            closeTag("redacteurs");
        }
        closeTag("entete");
    }

    private void addCorpsdefiche(FicheAPI fiche) throws IOException {
        List<Section> sectionList = fiche.getSectionList();
        if (sectionList.isEmpty()) {
            return;
        }
        openTag("corpsdefiche");
        for (Section section : sectionList) {
            addSection(section);
        }
        closeTag("corpsdefiche");

    }

    private void addProp(Prop prop) throws IOException {
        FicheItem ficheItem = prop.getFicheItem();
        startOpenTag("propriete");
        addAttribute("name", prop.getFieldKey().getFieldName());
        endOpenTag();
        ficheItemXMLPart.addFicheItem(ficheItem);
        closeTag("propriete");
    }

    private void addInfo(Info info) throws IOException {
        startOpenTag("information");
        addAttribute("name", info.getFieldKey().getFieldName());
        endOpenTag();
        ficheItemXMLPart.addFicheItems(info);
        closeTag("information");
    }

    private void addSection(Section section) throws IOException {
        startOpenTag("section");
        addAttribute("name", section.getFieldKey().getFieldName());
        endOpenTag();
        ficheBlockStorageXMLPart.addFicheBlocks(section);
        closeTag("section");
    }

}
