/* FichothequeLib_Tools - Copyright (c) 2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.tools.importation.thesaurus;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import net.fichotheque.importation.ThesaurusImport;
import net.fichotheque.thesaurus.Motcle;
import net.fichotheque.thesaurus.Thesaurus;
import net.fichotheque.tools.importation.ImportationUtils;


/**
 *
 * @author Vincent Calame
 */
public class EditThesaurusImportBuilder extends ThesaurusImportBuilder {

    private final List<BuilderInfo> builderList = new ArrayList<BuilderInfo>();
    private final Map<Integer, EditMotcleImportBuilder> creationByIdMap = new HashMap<Integer, EditMotcleImportBuilder>();
    private final Map<String, EditMotcleImportBuilder> creationByIdalphaMap = new HashMap<String, EditMotcleImportBuilder>();
    private final Map<Integer, EditMotcleImportBuilder> changeMap = new HashMap<Integer, EditMotcleImportBuilder>();

    public EditThesaurusImportBuilder(Thesaurus thesaurus) {
        super(thesaurus, ThesaurusImport.EDIT_TYPE);
    }

    public EditMotcleImportBuilder getEditMotcleImportBuilder(int newId, String newIdalpha) {
        EditMotcleImportBuilder builder;
        if (newId > 0) {
            builder = creationByIdMap.get(newId);
            if (builder == null) {
                newIdalpha = testIdalpha(newIdalpha);
                builder = EditMotcleImportBuilder.init(newId);
                add(builder, true);
                creationByIdMap.put(newId, builder);
                if (newIdalpha != null) {
                    builder.setNewIdalpha(newIdalpha);
                    creationByIdalphaMap.put(newIdalpha, builder);
                }
            }
        } else {
            newIdalpha = testIdalpha(newIdalpha);
            if (newIdalpha != null) {
                builder = creationByIdalphaMap.get(newIdalpha);
                if (builder == null) {
                    builder = EditMotcleImportBuilder.init(-1).setNewIdalpha(newIdalpha);
                    creationByIdalphaMap.put(newIdalpha, builder);
                    add(builder, true);
                }
            } else {
                builder = EditMotcleImportBuilder.init(-1);
                add(builder, true);
            }
        }
        return builder;
    }

    private String testIdalpha(String idalpha) {
        if (getThesaurus().isIdalphaType()) {
            if (idalpha == null) {
                throw new IllegalArgumentException("newIdalpha is null");
            }
            return idalpha;
        } else {
            return null;
        }
    }

    public EditMotcleImportBuilder getEditMotcleImportBuilder(Motcle motcle) {
        int motcleId = motcle.getId();
        EditMotcleImportBuilder builder = changeMap.get(motcleId);
        if (builder == null) {
            builder = new EditMotcleImportBuilder(motcleId);
            changeMap.put(motcle.getId(), builder);
            add(builder, false);
        }
        return builder;
    }

    private void add(EditMotcleImportBuilder builder, boolean isCreation) {
        builderList.add(new BuilderInfo(builder, isCreation));
    }

    @Override
    protected List<ThesaurusImport.MotcleImport> getMotcleImportList() {
        List<ThesaurusImport.MotcleImport> finalList = new ArrayList<ThesaurusImport.MotcleImport>();
        for (BuilderInfo builderInfo : builderList) {
            ThesaurusImport.EditMotcleImport motcleImport = builderInfo.builder.toEditMotcleImport();
            if ((builderInfo.isCreation) || !(isEmpty(motcleImport))) {
                finalList.add(motcleImport);
            }
        }
        return ImportationUtils.wrap(finalList.toArray(new ThesaurusImport.MotcleImport[finalList.size()]));
    }

    public static EditThesaurusImportBuilder init(Thesaurus thesaurus) {
        return new EditThesaurusImportBuilder(thesaurus);
    }

    private static boolean isEmpty(ThesaurusImport.EditMotcleImport editMotcleImport) {
        if (editMotcleImport.getNewIdalpha() != null) {
            return false;
        }
        if (editMotcleImport.getParentChange() != null) {
            return false;
        }
        if (editMotcleImport.getNewStatus() != null) {
            return false;
        }
        if (!editMotcleImport.getLabelChange().isEmpty()) {
            return false;
        }
        if (!editMotcleImport.getAttributeChange().isEmpty()) {
            return false;
        }
        if (!editMotcleImport.getLiensImport().isEmpty()) {
            return false;
        }
        return true;
    }


    private static class BuilderInfo {

        private final EditMotcleImportBuilder builder;
        private final boolean isCreation;

        private BuilderInfo(EditMotcleImportBuilder builder, boolean isCreation) {
            this.builder = builder;
            this.isCreation = isCreation;
        }

    }

}
