/* BdfServer - Copyright (c) 2021-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package fr.exemole.bdfserver.tools.exportation.table;

import fr.exemole.bdfserver.api.BdfServer;
import fr.exemole.bdfserver.tools.BdfServerUtils;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import net.fichotheque.SubsetItem;
import net.fichotheque.exportation.table.CellConverterProvider;
import net.fichotheque.exportation.table.SumCellConverter;
import net.fichotheque.exportation.table.TableExport;
import net.fichotheque.extraction.ExtractionContext;
import net.fichotheque.tools.exportation.table.SumCellConverterFactory;


/**
 *
 * @author Vincent Calame
 */
public final class CellConverterProviderFactory {

    private CellConverterProviderFactory() {

    }

    public static CellConverterProvider newInstance(BdfServer bdfServer, ExtractionContext extractionContext, Predicate<SubsetItem> globalPredicate, boolean withLangContextCheck) {
        return new InternalCellConverterProvider(bdfServer, extractionContext, withLangContextCheck, globalPredicate);
    }


    private static class InternalCellConverterProvider implements CellConverterProvider {

        private final BdfServer bdfServer;
        private final Map<String, SumCellConverter> engineMap = new HashMap<String, SumCellConverter>();
        private final Set<String> unknownSet = new HashSet<String>();
        private final ExtractionContext extractionContext;
        private final boolean withLangContextCheck;
        private final Predicate<SubsetItem> globalPredicate;

        private InternalCellConverterProvider(BdfServer bdfServer, ExtractionContext extractionContext, boolean withLangContextCheck, Predicate<SubsetItem> globalPredicate) {
            this.bdfServer = bdfServer;
            this.extractionContext = extractionContext;
            this.withLangContextCheck = withLangContextCheck;
            this.globalPredicate = globalPredicate;
        }

        @Override
        public SumCellConverter getCellConverter(String name) {
            SumCellConverter cellConverter = engineMap.get(name);
            if (cellConverter != null) {
                return cellConverter;
            }
            if (unknownSet.contains(name)) {
                return null;
            }
            TableExport tableExport = bdfServer.getTableExportManager().getTableExport(name);
            if (tableExport == null) {
                unknownSet.add(name);
                return null;
            }
            ExtractionContext customExtractionContext;
            if (withLangContextCheck) {
                customExtractionContext = BdfServerUtils.checkLangMode(extractionContext, bdfServer, tableExport.getTableExportDef());
            } else {
                customExtractionContext = extractionContext;
            }
            cellConverter = SumCellConverterFactory.newInstance(tableExport.getSubsetTableList(), bdfServer.getTableExportContext(), customExtractionContext, this, globalPredicate);
            engineMap.put(name, cellConverter);
            return cellConverter;
        }

    }

}
