/* global OpenDocument,Node */

/**
 * 
 * @constructor
 * @param {OpenDocument.CsvWriter} csvWriter
 * @param {Object} options
 */
OpenDocument.CsvConverter = function (csvWriter, options) {
    this.csvWriter = csvWriter;
    this.textDataAttribute = "text";
    this.csvHiddenFunctions = {
        row: null,
        cell: null,
        content: null
    };
    if (options) {
        if (options.textAlias) {
            this.textDataAttribute = options.textAlias;
        }
        if (options.functions) {
            if (options.functions.csvHidden) {
                for(let csvHiddenFunctionName in options.functions.csvHidden) {
                    this.csvHiddenFunctions[csvHiddenFunctionName] = options.functions.csvHidden[csvHiddenFunctionName];
                }
            }
        }
    }
};

OpenDocument.CsvConverter.convertToBlob = function (table, csvSettings, converterOptions) {
    var csv = OpenDocument.CsvConverter.convertToCsv(table, csvSettings, converterOptions);
    return new Blob([csv], {type: "text/csv" });
};

OpenDocument.CsvConverter.convertToCsv = function (table, csvSettings, converterOptions) {
    var csvWriter = new OpenDocument.CsvWriter(csvSettings);
    var csvConverter = new OpenDocument.CsvConverter(csvWriter, converterOptions);
    csvConverter.convert(table);
    return csvWriter.text;
};

OpenDocument.CsvConverter.prototype.convert = function (table) {
    if (typeof table === "string") {
        table = document.getElementById(table);
    }
    var converter = this;
    var cw = converter.csvWriter;
    var cellBuilderOptions = {
        textDataAttribute: converter.textDataAttribute,
        hiddenFunction: _getContentHiddenFunction()
    };
    var cellCounter = new OpenDocument.CellCounter();
    for(let row of table.rows) {
        if (_isCsvHidden(converter.csvHiddenFunctions.row, row)) {
            continue;
        }
        cellCounter.newRow();
        for(let cell of row.cells) {
            _addCell(cell);
        }
        cw.endRow();
    }
    
    function _addCell(cell) {
        if (_isCsvHidden(converter.csvHiddenFunctions.cell, cell)) {
            return;
        }
        let odCell = OpenDocument.Elements.Cell.build(cell, null, cellBuilderOptions);
        _updateCounter(odCell);
        cw.addCell(odCell);
    }
    
    
    function _isCsvHidden(csvHiddenFunction, element) {
        let display = window.getComputedStyle(element).display;
        let csvHidden;
        if (csvHiddenFunction) {
            csvHidden = csvHiddenFunction(element);
        }
        if ((!csvHidden) && (csvHidden !== 0)) {
            csvHidden = OpenDocument.checkCsvHidden(element);
        }
        if (((display === "none") && (csvHidden !== -1)) || (csvHidden === 1)) {
            return true;
        } else {
            return false;
        }
    }
    
    function _updateCounter(odCell) {
        for(let i = 0; i < odCell.emptyPrevious; i++) {
            cellCounter.newCell(1,1);
        }
        odCell.columnIndex = cellCounter.column - 1;
        odCell.jump = cellCounter.newCell(odCell.rowSpan, odCell.colSpan);
        for(let i = 0; i < odCell.emptyNext; i++) {
            cellCounter.newCell(1,1);
        }
    }
    
    function _getContentHiddenFunction() {
        if (converter.csvHiddenFunctions.content) {
            return converter.csvHiddenFunctions.content;
        } else {
            return OpenDocument.checkCsvHidden;
        }
    }
};