/* FichothequeLib_Tools - Copyright (c) 2013-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.tools.importation.parsers;

import java.text.ParseException;
import net.fichotheque.corpus.Corpus;
import net.fichotheque.corpus.FicheMeta;
import net.fichotheque.corpus.fiche.FicheItems;
import net.fichotheque.corpus.metadata.FieldKey;
import net.fichotheque.thesaurus.metadata.ThesaurusLangChecker;
import net.fichotheque.thesaurus.policies.PolicyProvider;
import net.fichotheque.tools.importation.ParseResultBuilder;
import net.fichotheque.tools.importation.corpus.EditCorpusImportBuilder;
import net.fichotheque.tools.importation.corpus.EditFicheImportBuilder;
import net.fichotheque.tools.importation.parsers.handlers.HandlerUtils;
import net.fichotheque.tools.parsers.FicheParser;
import net.mapeadores.util.primitives.FuzzyDate;


/**
 *
 * @author Vincent Calame
 */
public class CorpusCreationRowParser extends CorpusRowParser {

    private final EditCorpusImportBuilder editCorpusImportBuilder;
    private final ParseResultBuilder parseResultBuilder;
    private final FicheParser.Parameters ficheParserParameters;
    private final CorpusColumns corpusColumns;
    private final PolicyProvider policyProvider;
    private final ThesaurusLangChecker thesaurusLangChecker;
    private final FicheItems defaultOwners;

    CorpusCreationRowParser(EditCorpusImportBuilder editCorpusImportBuilder, ParseResultBuilder parseResultBuilder, FicheParser.Parameters ficheParserParameters, CorpusColumns corpusColumns, PolicyProvider policyProvider, ThesaurusLangChecker thesaurusLangChecker, FicheItems defaultOwners) {
        this.editCorpusImportBuilder = editCorpusImportBuilder;
        this.parseResultBuilder = parseResultBuilder;
        this.ficheParserParameters = ficheParserParameters;
        this.corpusColumns = corpusColumns;
        this.policyProvider = policyProvider;
        this.thesaurusLangChecker = thesaurusLangChecker;
        this.defaultOwners = defaultOwners;
    }

    @Override
    public EditCorpusImportBuilder getCorpusImportBuilder() {
        return editCorpusImportBuilder;
    }

    @Override
    public void parseRow(int rowNumber, Row row) {
        int idIndex = corpusColumns.getIdIndex();
        int columnCount = row.getColumnCount();
        Corpus corpus = editCorpusImportBuilder.getCorpus();
        int newId = -1;
        if ((idIndex != -1) && (columnCount > idIndex)) {
            String id = row.getColumnValue(idIndex).trim();
            try {
                newId = Integer.parseInt(id);
                FicheMeta ficheMeta = corpus.getFicheMetaById(newId);
                if (ficheMeta != null) {
                    parseResultBuilder.existingId(rowNumber, id);
                    return;
                }
            } catch (NumberFormatException nfe) {
                parseResultBuilder.notIntegerId(rowNumber, id);
                return;
            }
        }
        EditFicheImportBuilder ficheImportBuilder = editCorpusImportBuilder.getEditFicheImportBuilder(newId);
        if (corpusColumns.hasFicheHandler()) {
            HandlerUtils.populate(ficheImportBuilder.getFicheChangeBuilder(), corpusColumns.getFicheHandlerArray(), row, corpus, ficheParserParameters);
        }
        if (corpusColumns.hasAttributeHandler()) {
            HandlerUtils.populate(ficheImportBuilder.getAttributeChangeBuilder(), corpusColumns.getAttributeHandlerArray(), row);
        }
        if (corpusColumns.hasCroisementHandler()) {
            HandlerUtils.populate(ficheImportBuilder.getLiensImportBuilder(), corpusColumns.getCroisementHandlers(), row, corpus, ficheParserParameters.getWorkingLang(), policyProvider, thesaurusLangChecker);
        }
        if ((defaultOwners != null) && (!corpusColumns.hasColumn(FieldKey.OWNERS))) {
            ficheImportBuilder.getFicheChangeBuilder().putOwners(defaultOwners);
        }
        int creationDateIndex = corpusColumns.getCreationDateIndex();
        if ((creationDateIndex != -1) && (columnCount > creationDateIndex)) {
            String creationDateString = row.getColumnValue(creationDateIndex).trim();
            try {
                FuzzyDate creationDate = FuzzyDate.parse(creationDateString);
                ficheImportBuilder.setCreationDate(creationDate);
            } catch (ParseException die) {
                parseResultBuilder.wrongDate(rowNumber, creationDateString);
            }
        }
    }

    static CorpusCreationRowParser newInstance(String[] fieldArray, EditCorpusImportBuilder editCorpusImportBuilder, ParseResultBuilder parseResultBuilder, FicheParser.Parameters ficheParserParameters, PolicyProvider policyProvider, ThesaurusLangChecker thesaurusLangChecker, FicheItems defaultOwners) {
        CorpusColumns corpusColumns = CorpusColumns.parse(fieldArray, editCorpusImportBuilder, parseResultBuilder);
        return new CorpusCreationRowParser(editCorpusImportBuilder, parseResultBuilder, ficheParserParameters, corpusColumns, policyProvider, thesaurusLangChecker, defaultOwners);
    }

}
