/* FichothequeLib_Tools  - Copyright (c) 2007-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.tools.format.formatters;

import java.io.IOException;
import java.util.List;
import net.fichotheque.addenda.Document;
import net.fichotheque.addenda.Version;
import net.fichotheque.format.FormatContext;
import net.fichotheque.format.FormatSource;
import net.fichotheque.format.formatters.DocumentFormatter;
import net.fichotheque.json.DocumentJson;
import net.fichotheque.tools.format.patterndefs.DefaultPattern;
import net.fichotheque.utils.FormatterUtils;
import net.mapeadores.util.exceptions.ShouldNotOccurException;
import net.mapeadores.util.format.FormatConstants;
import net.mapeadores.util.instruction.Argument;
import net.mapeadores.util.instruction.Instruction;
import net.mapeadores.util.instruction.InstructionResolver;
import net.mapeadores.util.instruction.InstructionResolverProvider;
import net.mapeadores.util.json.JSONWriter;
import net.mapeadores.util.localisation.LangContext;
import net.mapeadores.util.logging.ErrorMessageException;
import net.mapeadores.util.logging.MessageHandler;


/**
 *
 * @author Vincent Calame
 */
public final class DocumentFormatterParser {

    private final static DocumentFormatter ADDENDA_PART = (document, weight, formatSource) -> {
        return document.getSubsetName();
    };
    private final static DocumentFormatter BASENAME_PART = (document, weight, formatSource) -> {
        return document.getBasename();
    };
    private final static DocumentFormatter EXTENSIONS_PART = (document, weight, formatSource) -> {
        List<Version> versionList = document.getVersionList();
        if (versionList.size() == 1) {
            return '.' + versionList.get(0).getExtension();
        } else {
            StringBuilder buf = new StringBuilder();
            for (Version version : versionList) {
                buf.append('[');
                buf.append('.');
                buf.append(version.getExtension());
                buf.append(']');
            }
            return buf.toString();
        }
    };
    private final static DocumentFormatter ID_PART = (document, weight, formatSource) -> {
        return String.valueOf(document.getId());
    };
    private final static DocumentFormatter WEIGHT_PART = (document, weight, formatSource) -> {
        return String.valueOf(weight);
    };
    private final static DocumentFormatter WITHWEIGHTFILTER_FORMSYNTAX_PART = new FormSyntaxPart(true);
    private final static DocumentFormatter WITHOUTWEIGHTFILTER_FORMSYNTAX_PART = new FormSyntaxPart(false);
    private final static JsonParameters DEFAULT_JSONPARAMETERS = JsonParameters.parse("basename,versions,size");

    private DocumentFormatterParser() {
    }

    public static DocumentFormatter parse(String pattern, FormatContext formatContext, boolean withWeightFilter, MessageHandler messageHandler) {
        if (pattern == null) {
            pattern = DefaultPattern.document();
        }
        try {
            return parse(pattern, formatContext, withWeightFilter);
        } catch (ErrorMessageException fe) {
            messageHandler.addMessage(FormatConstants.SEVERE_PATTERN, fe.getErrorMessage());
            return null;
        }
    }

    public static DocumentFormatter parse(String pattern, FormatContext formatContext, boolean withWeightFilter) throws ErrorMessageException {
        Object[] partArray = FormatterUtils.parsePattern(new InternalInstructionResolver(formatContext, withWeightFilter), pattern);
        return new InternalDocumentFormatter(partArray);
    }


    private static class InternalDocumentFormatter implements DocumentFormatter {

        private final Object[] partArray;

        private InternalDocumentFormatter(Object[] partArray) {
            this.partArray = partArray;
        }

        @Override
        public String formatDocument(Document document, int weight, FormatSource formatSource) {
            StringBuilder buf = new StringBuilder();
            int length = partArray.length;
            for (int i = 0; i < length; i++) {
                Object obj = partArray[i];
                if (obj instanceof String) {
                    buf.append((String) obj);
                    continue;
                }
                DocumentFormatter formatter = (DocumentFormatter) obj;
                buf.append(formatter.formatDocument(document, weight, formatSource));
            }
            return buf.toString();
        }

    }


    private static class InternalInstructionResolver implements InstructionResolver {

        private final InstructionResolverProvider provider;
        private final boolean withWeightFilter;

        private InternalInstructionResolver(FormatContext formatContext, boolean withWeightFilter) {
            this.provider = formatContext.getInstructionResolverProvider();
            this.withWeightFilter = withWeightFilter;
        }

        @Override
        public Object resolve(Instruction instruction) throws ErrorMessageException {
            InstructionResolver resolver = provider.getInstructionResolver(DocumentFormatter.class);
            if (resolver != null) {
                Object formatter = resolver.resolve(instruction);
                if (formatter != null) {
                    return formatter;
                }
            }
            Argument arg1 = instruction.get(0);
            String key = arg1.getKey();
            switch (key) {
                case "addenda":
                    return ADDENDA_PART;
                case "basename":
                    return BASENAME_PART;
                case "extensions":
                    return EXTENSIONS_PART;
                case "formsyntax":
                    if (withWeightFilter) {
                        return WITHWEIGHTFILTER_FORMSYNTAX_PART;
                    } else {
                        return WITHOUTWEIGHTFILTER_FORMSYNTAX_PART;
                    }
                case "id":
                    return ID_PART;
                case "json": {
                    JsonParameters jsonParameters = JsonParameters.fromInstruction(instruction);
                    if (jsonParameters == null) {
                        jsonParameters = DEFAULT_JSONPARAMETERS;
                    }
                    LangParameters langParameters = LangParameters.fromInstruction(instruction, "langs");
                    return new JsonPart(jsonParameters, langParameters.getCustomLangContext());
                }
                case "weight":
                case "poids":
                    return WEIGHT_PART;
                default:
                    return null;

            }
        }

    }


    private static class JsonPart implements DocumentFormatter {

        private final JsonParameters jsonParameters;
        private final LangContext customLangContext;

        private JsonPart(JsonParameters jsonParameters, LangContext customLangContext) {
            this.jsonParameters = jsonParameters;
            this.customLangContext = customLangContext;
        }

        @Override
        public String formatDocument(Document document, int weight, FormatSource formatSource) {
            LangContext langContext;
            if (customLangContext != null) {
                langContext = customLangContext;
            } else {
                langContext = formatSource.getLangContext();
            }
            StringBuilder buf = new StringBuilder();
            JSONWriter jsonWriter = new JSONWriter(buf);
            try {
                jsonWriter.object();
                DocumentJson.properties(jsonWriter, document, langContext, jsonParameters);
                jsonWriter.endObject();
            } catch (IOException ioe) {
                throw new ShouldNotOccurException(ioe);
            }
            return buf.toString();
        }

    }


    private static class FormSyntaxPart implements DocumentFormatter {

        private final boolean withWeightFilter;

        private FormSyntaxPart(boolean withWeightFilter) {
            this.withWeightFilter = withWeightFilter;
        }

        @Override
        public String formatDocument(Document document, int weight, FormatSource formatSource) {
            if ((withWeightFilter) || (weight == 1)) {
                return String.valueOf(document.getId());
            } else {
                StringBuilder buf = new StringBuilder();
                buf.append(String.valueOf(document.getId()));
                buf.append(" <");
                buf.append(weight);
                buf.append(">");
                return buf.toString();
            }
        }

    }


}
