/* FichothequeLib_Xml - Copyright (c) 2013-2024 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.xml.importation;

import java.io.IOException;
import net.fichotheque.importation.ParentChange;
import net.fichotheque.importation.ThesaurusImport;
import net.fichotheque.thesaurus.Thesaurus;
import net.mapeadores.util.attr.AttributeChange;
import net.mapeadores.util.attr.AttributeKey;
import net.mapeadores.util.attr.AttributeUtils;
import net.mapeadores.util.localisation.Lang;
import net.mapeadores.util.text.LabelChange;
import net.mapeadores.util.text.LabelUtils;
import net.mapeadores.util.xml.XMLPart;
import net.mapeadores.util.xml.XMLWriter;


/**
 *
 * @author Vincent Calame
 */
public class ThesaurusImportXMLPart extends XMLPart {

    private final LiensImportXMLPart liensImportXMLPart;

    public ThesaurusImportXMLPart(XMLWriter xmlWriter) {
        super(xmlWriter);
        this.liensImportXMLPart = new LiensImportXMLPart(xmlWriter);
    }

    public void addThesaurusImport(ThesaurusImport thesaurusImport) throws IOException {
        String type = thesaurusImport.getType();
        startOpenTag("thesaurusimport");
        addAttribute("thesaurus", thesaurusImport.getThesaurus().getSubsetName());
        addAttribute("type", type);
        Thesaurus destinationThesaurus = thesaurusImport.getDestinationThesaurus();
        if (destinationThesaurus != null) {
            addAttribute("destination", destinationThesaurus.getSubsetName());
        }
        endOpenTag();
        for (ThesaurusImport.MotcleImport motcleImport : thesaurusImport.getMotcleImportList()) {
            if (motcleImport instanceof ThesaurusImport.EditMotcleImport) {
                addEditMotcleImport((ThesaurusImport.EditMotcleImport) motcleImport);
            } else if (motcleImport instanceof ThesaurusImport.MergeMotcleImport) {
                addMergeMotcleImport((ThesaurusImport.MergeMotcleImport) motcleImport);
            } else {
                addMotcleImport(motcleImport);
            }
        }
        closeTag("thesaurusimport");
    }

    private void addMotcleImport(ThesaurusImport.MotcleImport motcleImport) throws IOException {
        startOpenTag("motcleimport");
        addAttribute("id", motcleImport.getMotcleId());
        closeEmptyTag();
    }

    private void addEditMotcleImport(ThesaurusImport.EditMotcleImport motcleImport) throws IOException {
        startOpenTag("motcleimport");
        int id = motcleImport.getMotcleId();
        if (id > 0) {
            addAttribute("id", id);
        }
        endOpenTag();
        String newIdalpha = motcleImport.getNewIdalpha();
        if (newIdalpha != null) {
            addSimpleElement("idalpha", newIdalpha);
        }
        String newStatus = motcleImport.getNewStatus();
        if (newStatus != null) {
            addSimpleElement("status", newStatus);
        }
        ParentChange parentChange = motcleImport.getParentChange();
        if (parentChange != null) {
            startOpenTag("parent");
            if (parentChange instanceof ParentChange.Id) {
                addAttribute("id", ((ParentChange.Id) parentChange).getParentId());
            } else if (parentChange instanceof ParentChange.Idalpha) {
                addAttribute("idalpha", ((ParentChange.Idalpha) parentChange).getParentIdalpha());
            }
            closeEmptyTag();
        }
        addLabelChange(motcleImport.getLabelChange());
        addAttributeChange(motcleImport.getAttributeChange());
        liensImportXMLPart.addLiensImport(motcleImport.getLiensImport());
        closeTag("motcleimport");
    }

    private void addMergeMotcleImport(ThesaurusImport.MergeMotcleImport motcleImport) throws IOException {
        startOpenTag("motcleimport");
        addAttribute("id", motcleImport.getMotcleId());
        addAttribute("destination", motcleImport.getDestinationMotcle().getId());
        closeEmptyTag();
    }

    private void addLabelChange(LabelChange labelChange) throws IOException {
        LabelUtils.addLabels(this, labelChange.getChangedLabels());
        for (Lang lang : labelChange.getRemovedLangList()) {
            startOpenTag("label");
            addAttribute("xml:lang", lang.toString());
            closeEmptyTag();
        }
    }

    private void addAttributeChange(AttributeChange attributeChange) throws IOException {
        AttributeUtils.addAttributes(this, attributeChange.getChangedAttributes());
        for (AttributeKey attributeKey : attributeChange.getRemovedAttributeKeyList()) {
            startOpenTag("attr");
            addAttribute("ns", attributeKey.getNameSpace());
            addAttribute("key", attributeKey.getLocalKey());
            closeEmptyTag();
        }
    }

}
