/* FichothequeLib_Tools - Copyright (c) 2013-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.tools.importation;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.RandomAccess;
import net.fichotheque.importation.ParseResult;


/**
 *
 * @author Vincent Calame
 */
public class ParseResultBuilder {

    private final String type;
    private final List<ParseResult.InitError> initErrorList = new ArrayList<ParseResult.InitError>();
    private final List<ParseResult.InitWarning> initWarningList = new ArrayList<ParseResult.InitWarning>();
    private final List<ParseResult.ParseError> parseErrorList = new ArrayList<ParseResult.ParseError>();
    private final List<ParseResult.BdfError> bdfErrorList = new ArrayList<ParseResult.BdfError>();
    private final Map<String, ParseResult.ResultItem> resultItemMap = new LinkedHashMap<String, ParseResult.ResultItem>();
    private String path = "";

    public ParseResultBuilder(String type) {
        this.type = type;
    }

    public ParseResultBuilder addInitError(String key, String value) {
        initErrorList.add(new InternalInitError(key, value));
        return this;
    }

    public ParseResultBuilder addInitWarning(String key, String value) {
        initWarningList.add(new InternalInitWarning(key, value));
        return this;
    }

    public ParseResultBuilder addParseError(String parseError, int lineNumber) {
        parseErrorList.add(new InternalParseError(parseError, lineNumber));
        return this;
    }

    public ParseResultBuilder addBdfError(String key, String text, int lineNumber) {
        bdfErrorList.add(new InternalBdfError(key, text, lineNumber));
        return this;
    }

    public boolean checkDoublon(String idString, int lineNumber) {
        ParseResult.ResultItem currentResultItem = resultItemMap.get(idString);
        if (currentResultItem != null) {
            addBdfError("DOUBLON", String.valueOf(currentResultItem.getLineNumber()), lineNumber);
            return true;
        } else {
            return false;
        }
    }

    public ParseResultBuilder addResult(String idString, Object key, Object value, int lineNumber) {
        ParseResult.ResultItem resultItem = new InternalResultItem(key, value, lineNumber);
        resultItemMap.put(idString, resultItem);
        return this;
    }

    public ParseResultBuilder setPath(String path) {
        this.path = path;
        return this;
    }

    public ParseResult toParseResult() {
        List<ParseResult.InitError> finalInitErrorList = new InitErrorList(initErrorList.toArray(new ParseResult.InitError[initErrorList.size()]));
        List<ParseResult.InitWarning> finalInitWarningList = new InitWarningList(initWarningList.toArray(new ParseResult.InitWarning[initWarningList.size()]));
        List<ParseResult.ParseError> finalParseErrorList = new ParseErrorList(parseErrorList.toArray(new ParseResult.ParseError[parseErrorList.size()]));
        List<ParseResult.BdfError> finalBdfErrorList = new BdfErrorList(bdfErrorList.toArray(new ParseResult.BdfError[bdfErrorList.size()]));
        List<ParseResult.ResultItem> resultItemList = new ResultItemList(resultItemMap.values().toArray(new ParseResult.ResultItem[resultItemMap.size()]));
        return new InternalParseResult(type, finalInitErrorList, finalInitWarningList, finalParseErrorList, finalBdfErrorList, resultItemList, path);
    }

    public static ParseResultBuilder init(String type) {
        return new ParseResultBuilder(type);
    }


    private static class InternalParseResult implements ParseResult {

        private final String type;
        private final List<ParseResult.InitError> initErrorList;
        private final List<ParseResult.InitWarning> initWarningList;
        private final List<ParseResult.ParseError> parseErrorList;
        private final List<ParseResult.BdfError> bdfErrorList;
        private final List<ParseResult.ResultItem> resultItemList;
        private final String path;

        private InternalParseResult(String type, List<ParseResult.InitError> initErrorList, List<ParseResult.InitWarning> initWarningList, List<ParseResult.ParseError> parseErrorList, List<ParseResult.BdfError> bdfErrorList, List<ParseResult.ResultItem> resultItemList, String path) {
            this.type = type;
            this.initErrorList = initErrorList;
            this.initWarningList = initWarningList;
            this.parseErrorList = parseErrorList;
            this.bdfErrorList = bdfErrorList;
            this.resultItemList = resultItemList;
            this.path = path;
        }

        @Override
        public String getType() {
            return type;
        }

        @Override
        public List<ParseResult.InitError> getInitErrorList() {
            return initErrorList;
        }

        @Override
        public List<ParseResult.InitWarning> getInitWarningList() {
            return initWarningList;
        }

        @Override
        public List<ParseResult.ParseError> getParseErrorList() {
            return parseErrorList;
        }

        @Override
        public List<ParseResult.BdfError> getBdfErrorList() {
            return bdfErrorList;
        }

        @Override
        public String getPath() {
            return path;
        }

        @Override
        public List<ResultItem> getResultItemList() {
            return resultItemList;
        }

    }


    private static class InternalInitError implements ParseResult.InitError {

        private final String key;
        private final String text;

        private InternalInitError(String key, String text) {
            this.key = key;
            this.text = text;
        }

        @Override
        public String getKey() {
            return key;
        }

        @Override
        public String getText() {
            return text;
        }

    }


    private static class InternalInitWarning implements ParseResult.InitWarning {

        private final String key;
        private final String text;

        private InternalInitWarning(String key, String text) {
            this.key = key;
            this.text = text;
        }

        @Override
        public String getKey() {
            return key;
        }

        @Override
        public String getText() {
            return text;
        }

    }


    private static class InternalParseError implements ParseResult.ParseError {

        private final String rawText;
        private final int lineNumber;

        private InternalParseError(String rawText, int lineNumber) {
            this.rawText = rawText;
            this.lineNumber = lineNumber;
        }

        @Override
        public String getRawText() {
            return rawText;
        }

        @Override
        public int getLineNumber() {
            return lineNumber;
        }

    }


    private static class InternalBdfError implements ParseResult.BdfError {

        private final String key;
        private final String text;
        private final int lineNumber;

        private InternalBdfError(String key, String text, int lineNumber) {
            this.key = key;
            this.text = text;
            this.lineNumber = lineNumber;
        }

        @Override
        public String getKey() {
            return key;
        }

        @Override
        public String getText() {
            return text;
        }

        @Override
        public int getLineNumber() {
            return lineNumber;
        }

    }


    private static class InternalResultItem implements ParseResult.ResultItem {

        private final Object keyObject;
        private final Object valueObject;
        private final int lineNumber;

        private InternalResultItem(Object keyObject, Object valueObject, int lineNumber) {
            this.keyObject = keyObject;
            this.valueObject = valueObject;
            this.lineNumber = lineNumber;
        }

        @Override
        public Object getKeyObject() {
            return keyObject;
        }

        @Override
        public Object getValueObject() {
            return valueObject;
        }

        @Override
        public int getLineNumber() {
            return lineNumber;
        }

    }


    private static class ResultItemList extends AbstractList<ParseResult.ResultItem> implements RandomAccess {

        private final ParseResult.ResultItem[] array;

        private ResultItemList(ParseResult.ResultItem[] array) {
            this.array = array;
        }

        @Override
        public int size() {
            return array.length;
        }

        @Override
        public ParseResult.ResultItem get(int index) {
            return array[index];
        }

    }


    private static class InitErrorList extends AbstractList<ParseResult.InitError> implements RandomAccess {

        private final ParseResult.InitError[] array;

        private InitErrorList(ParseResult.InitError[] array) {
            this.array = array;
        }

        @Override
        public int size() {
            return array.length;
        }

        @Override
        public ParseResult.InitError get(int index) {
            return array[index];
        }

    }


    private static class InitWarningList extends AbstractList<ParseResult.InitWarning> implements RandomAccess {

        private final ParseResult.InitWarning[] array;

        private InitWarningList(ParseResult.InitWarning[] array) {
            this.array = array;
        }

        @Override
        public int size() {
            return array.length;
        }

        @Override
        public ParseResult.InitWarning get(int index) {
            return array[index];
        }

    }


    private static class ParseErrorList extends AbstractList<ParseResult.ParseError> implements RandomAccess {

        private final ParseResult.ParseError[] array;

        private ParseErrorList(ParseResult.ParseError[] array) {
            this.array = array;
        }

        @Override
        public int size() {
            return array.length;
        }

        @Override
        public ParseResult.ParseError get(int index) {
            return array[index];
        }

    }


    private static class BdfErrorList extends AbstractList<ParseResult.BdfError> implements RandomAccess {

        private final ParseResult.BdfError[] array;

        private BdfErrorList(ParseResult.BdfError[] array) {
            this.array = array;
        }

        @Override
        public int size() {
            return array.length;
        }

        @Override
        public ParseResult.BdfError get(int index) {
            return array[index];
        }

    }

}
