/* FichothequeLib_Tools - Copyright (c) 2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.tools.format.catalogs;

import java.io.IOException;
import net.fichotheque.exportation.table.Cell;
import net.fichotheque.format.FormatSource;
import net.fichotheque.format.formatters.MotcleFormatter;
import net.fichotheque.json.MotcleJson;
import net.fichotheque.thesaurus.Motcle;
import net.fichotheque.tools.format.FormatterEngineUtils;
import net.fichotheque.tools.format.JsonParameters;
import net.fichotheque.tools.format.LangParameters;
import net.fichotheque.utils.ThesaurusUtils;
import net.mapeadores.util.exceptions.ShouldNotOccurException;
import net.mapeadores.util.json.JSONWriter;
import net.mapeadores.util.localisation.Lang;
import net.mapeadores.util.localisation.LangContext;


/**
 *
 * @author Vincent Calame
 */
public final class MotcleFormatterCatalog {

    public final static MotcleFormatter ID = (motcle, weight, formatSource) -> {
        return String.valueOf(motcle.getId());
    };
    public final static MotcleFormatter IDALPHA = (motcle, weight, formatSource) -> {
        String idalpha = motcle.getIdalpha();
        if (idalpha == null) {
            idalpha = "";
        }
        return idalpha;
    };
    public final static MotcleFormatter IDALPHA_SIGNIFICANT = (motcle, weight, formatSource) -> {
        String idalpha = motcle.getSignificantIdalpha();
        if (idalpha == null) {
            idalpha = "";
        }
        return idalpha;
    };
    public final static MotcleFormatter LANG = (motcle, weight, formatSource) -> {
        if (!motcle.isBabelienType()) {
            return "";
        }
        net.mapeadores.util.text.Label label = motcle.getBabelienLabel();
        return label.getLang().toString();
    };
    public final static MotcleFormatter WEIGHT = (motcle, weight, formatSource) -> {
        return String.valueOf(weight);
    };
    public final static MotcleFormatter PARENT_ID = (motcle, weight, formatSource) -> {
        Motcle parent = motcle.getParent();
        if (parent != null) {
            return String.valueOf(parent.getId());
        } else {
            return "";
        }
    };
    public final static MotcleFormatter PARENT_IDALPHA = (motcle, weight, formatSource) -> {
        String idalpha = null;
        Motcle parentMotcle = motcle.getParent();
        if (parentMotcle != null) {
            idalpha = parentMotcle.getIdalpha();
        }
        if (idalpha == null) {
            idalpha = "";
        }
        return idalpha;
    };
    public final static MotcleFormatter PARENT_IDALPHA_SIGNIFICANT = (motcle, weight, formatSource) -> {
        String idalpha = null;
        Motcle parentMotcle = motcle.getParent();
        if (parentMotcle != null) {
            idalpha = parentMotcle.getSignificantIdalpha();
        }
        if (idalpha == null) {
            idalpha = "";
        }
        return idalpha;
    };
    public final static MotcleFormatter POSITION = (motcle, weight, formatSource) -> {
        return String.valueOf(motcle.getChildIndex() + 1);
    };
    public final static MotcleFormatter POSITION_GLOBAL = (motcle, weight, formatSource) -> {
        return ThesaurusUtils.getGlobalPosition(motcle);
    };
    public final static MotcleFormatter THESAURUS = (motcle, weight, formatSource) -> {
        return motcle.getSubsetName();
    };
    public final static MotcleFormatter WITHWEIGHTFILTER_FORMSYNTAX_FORMATTER = new InternalFormSyntax(true);
    public final static MotcleFormatter WITHOUTWEIGHTFILTER_FORMSYNTAX_FORMATTER = new InternalFormSyntax(false);

    private MotcleFormatterCatalog() {

    }


    public static class Json implements MotcleFormatter {

        private final JsonParameters jsonParameters;
        private final LangContext customLangContext;

        public Json(JsonParameters jsonParameters, LangContext customLangContext) {
            this.jsonParameters = jsonParameters;
            this.customLangContext = customLangContext;
        }

        @Override
        public String formatMotcle(Motcle motcle, int weight, FormatSource formatSource) {
            Cell[] cellArray = FormatterEngineUtils.resolve(motcle, jsonParameters, formatSource);
            LangContext langContext;
            if (customLangContext != null) {
                langContext = customLangContext;
            } else {
                langContext = formatSource.getLangContext();
            }
            StringBuilder buf = new StringBuilder();
            JSONWriter jsonWriter = new JSONWriter(buf);
            try {
                jsonWriter.object();
                MotcleJson.properties(jsonWriter, motcle, langContext, jsonParameters.getPropertyEligibility(), cellArray);
                jsonWriter.endObject();
            } catch (IOException ioe) {
                throw new ShouldNotOccurException(ioe);
            }
            return buf.toString();
        }

    }


    public static class Label implements MotcleFormatter {

        private final LangParameters langParameters;

        public Label(LangParameters langParameters) {
            this.langParameters = langParameters;
        }

        @Override
        public String formatMotcle(Motcle motcle, int weight, FormatSource formatSource) {
            StringBuilder buf = new StringBuilder();
            Lang[] array = LangParameters.checkLangArray(langParameters, formatSource);
            for (Lang lang : array) {
                net.mapeadores.util.text.Label label = motcle.getLabels().getLangPartCheckedLabel(lang);
                if (label != null) {
                    if (buf.length() > 0) {
                        buf.append(langParameters.getSeparator());
                    }
                    buf.append(label.getLabelString());
                }
            }
            return buf.toString();
        }

    }


    private static class InternalFormSyntax implements MotcleFormatter {

        private final boolean withWeightFilter;

        private InternalFormSyntax(boolean withWeightFilter) {
            this.withWeightFilter = withWeightFilter;
        }

        @Override
        public String formatMotcle(Motcle motcle, int weight, FormatSource formatSource) {
            if ((withWeightFilter) || (weight == 1)) {
                String idalpha = motcle.getIdalpha();
                if (idalpha != null) {
                    return idalpha;
                } else {
                    return String.valueOf(motcle.getId());
                }
            } else {
                StringBuilder buf = new StringBuilder();
                String idalpha = motcle.getIdalpha();
                if (idalpha != null) {
                    buf.append(idalpha);
                } else {
                    buf.append(String.valueOf(motcle.getId()));
                }
                buf.append(" <");
                buf.append(weight);
                buf.append(">");
                return buf.toString();
            }
        }

    }

}
