/* FichothequeLib_API - Copyright (c) 2013-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.utils.selection;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import net.fichotheque.junction.JunctionKey;
import net.fichotheque.junction.Tie;
import net.fichotheque.selection.RangeCondition;
import net.fichotheque.utils.JunctionUtils;
import net.mapeadores.util.attr.Attributes;
import net.fichotheque.junction.Junction;
import net.fichotheque.selection.JunctionCondition;


/**
 *
 * @author Vincent Calame
 */
class FilteredJunctionBuilder {

    private final Junction source;
    private final Map<String, Tie> tieMap = new LinkedHashMap<String, Tie>();

    FilteredJunctionBuilder(Junction source) {
        this.source = source;
    }

    void testCondition(JunctionCondition junctionCondition) {
        if (junctionCondition == null) {
            for (Tie tie : source.getTieList()) {
                addTie(tie);
            }
            return;
        }
        List<String> modeList = junctionCondition.getTieModeList();
        RangeCondition weightCondition = junctionCondition.getWeightRangeCondition();
        if (weightCondition != null) {
            if (modeList.isEmpty()) {
                for (Tie tie : source.getTieList()) {
                    if (weightCondition.accept(tie.getWeight())) {
                        addTie(tie);
                    }
                }
            } else {
                for (String mode : modeList) {
                    Tie tie = source.getTieByMode(mode);
                    if ((tie != null) && (weightCondition.accept(tie.getWeight()))) {
                        addTie(tie);
                    }
                }
            }
        } else {
            for (String mode : modeList) {
                Tie tie = source.getTieByMode(mode);
                if (tie != null) {
                    addTie(tie);
                }
            }
        }
    }

    void addTie(Tie tie) {
        tieMap.put(tie.getMode(), tie);
    }

    Junction toJunction() {
        int size = tieMap.size();
        if (size == 0) {
            return null;
        }
        return new InternalFilteredJunction(source, tieMap.values().toArray(new Tie[size]));
    }


    private static class InternalFilteredJunction implements Junction {

        private final Junction source;
        private final Tie[] tieArray;
        private final List<Tie> tieList;

        private InternalFilteredJunction(Junction source, Tie[] tieArray) {
            this.source = source;
            this.tieArray = tieArray;
            this.tieList = JunctionUtils.wrap(tieArray);
        }

        @Override
        public JunctionKey getJunctionKey() {
            return source.getJunctionKey();
        }

        @Override
        public List<Tie> getTieList() {
            return tieList;
        }

        @Override
        public Tie getTieByMode(String mode) {
            for (Tie tie : tieArray) {
                if (tie.getMode().equals(mode)) {
                    return tie;
                }
            }
            return null;
        }

        @Override
        public Attributes getAttributes() {
            return source.getAttributes();
        }

    }

}
