/* BdfServer_HtmlProducers - Copyright (c) 2012-2026 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package fr.exemole.bdfserver.htmlproducers.album;

import fr.exemole.bdfserver.api.instruction.BdfInstructionConstants;
import fr.exemole.bdfserver.api.instruction.OutputParameters;
import fr.exemole.bdfserver.api.interaction.domains.AlbumDomain;
import fr.exemole.bdfserver.html.BdfServerHtmlProducer;
import fr.exemole.bdfserver.tools.instruction.BdfErrors;
import fr.exemole.bdfserver.tools.instruction.OutputRequestHandler;
import fr.exemole.bdfserver.tools.instruction.PermissionChecker;
import java.io.File;
import net.fichotheque.album.Album;
import net.fichotheque.album.Illustration;
import net.mapeadores.util.logging.ErrorMessageException;
import net.mapeadores.util.request.RequestMap;


/**
 *
 * @author Vincent Calame
 */
public final class AlbumHtmlProducerFactory {

    private final static int ALBUM_ADMIN = 1;
    private final static int ALBUM_READ = 2;
    private final static int ILLUSTRATION = 3;

    private AlbumHtmlProducerFactory() {
    }

    public static BdfServerHtmlProducer getHtmlProducer(OutputParameters parameters) throws ErrorMessageException {
        OutputRequestHandler requestHandler = OutputRequestHandler.init(parameters);
        String page = parameters.getOutput();
        RequestMap requestMap = parameters.getRequestMap();
        PermissionChecker permissionChecker = requestHandler.getPermissionChecker();
        switch (getPageType(page)) {
            case ALBUM_ADMIN: {
                Album album = requestHandler.getAlbum();
                permissionChecker.checkSubsetAdmin(album);
                switch (page) {
                    case AlbumDomain.ALBUM_ADVANCEDCOMMANDS_PAGE: {
                        return new AlbumAdvancedCommandsHtmlProducer(parameters, album);
                    }
                    case AlbumDomain.ALBUM_METADATAFORM_PAGE: {
                        return new AlbumMetadataFormHtmlProducer(parameters, album);
                    }
                    default:
                        return null;
                }
            }
            case ALBUM_READ: {
                Album album = requestHandler.getAlbum();
                switch (page) {
                    case AlbumDomain.ALBUM_PAGE: {
                        return new AlbumHtmlProducer(parameters, album);
                    }
                    case AlbumDomain.ILLUSTRATION_UPLOAD_NEW_PAGE: {
                        String callKey = requestHandler.getCallKey();
                        return IllustrationUploadHtmlProducer.newNewUpload(parameters, album, callKey);
                    }
                    case AlbumDomain.ILLUSTRATION_UPLOAD_REPLACE_PAGE: {
                        String callKey = requestHandler.getCallKey();
                        return IllustrationUploadHtmlProducer.newReplaceUpload(parameters, album, callKey);
                    }
                    case AlbumDomain.ILLUSTRATION_CHANGE_PAGE: {
                        String callKey = requestHandler.getCallKey();
                        File tmpFile = (File) parameters.getResultObject(BdfInstructionConstants.TMPFILE_OBJ);
                        if (tmpFile != null) {
                            return IllustrationChangeHtmlProducer.newTmpFileCreate(parameters, album, callKey, tmpFile.getName());
                        }
                        String changeType = requestHandler.getMandatoryParameter("changetype");
                        if (changeType.equals(AlbumDomain.TMPFILE_CHANGETYPE)) {
                            String tmpFileName = requestHandler.getMandatoryParameter("tmpfile");
                            return IllustrationChangeHtmlProducer.newTmpFileChange(parameters, album, callKey, tmpFileName);
                        } else if (changeType.equals(AlbumDomain.ILLUSTRATION_CHANGETYPE)) {
                            Illustration illustration = (Illustration) requestHandler.getMandatorySubsetItem(album);
                            String tmpFileName = requestMap.getParameter("tmpfile");
                            return IllustrationChangeHtmlProducer.newIllustrationChange(parameters, album, callKey, illustration, tmpFileName);
                        } else {
                            throw BdfErrors.unknownParameterValue("changetype", changeType);
                        }
                    }
                    default:
                        return null;
                }
            }
            case ILLUSTRATION: {
                Illustration illustration = requestHandler.getIllustration();
                switch (page) {
                    case AlbumDomain.ILLUSTRATION_ADMINFORM_PAGE: {
                        permissionChecker.checkWrite(illustration);
                        return new IllustrationAdminFormHtmlProducer(parameters, illustration);
                    }
                    case AlbumDomain.ILLUSTRATION_ADVANCEDCOMMANDS_PAGE: {
                        permissionChecker.checkSubsetAdmin(illustration.getAlbum());
                        return new IllustrationAdvancedCommandsHtmlProducer(parameters, illustration);
                    }
                    default:
                        return null;
                }
            }
            default: {
                switch (page) {
                    case AlbumDomain.ALBUM_CREATIONFORM_PAGE: {
                        permissionChecker.checkFichothequeAdmin();
                        return new AlbumCreationFormHtmlProducer(parameters);
                    }
                    case AlbumDomain.ILLUSTRATION_UPLOAD_CONFIRM_PAGE: {
                        String callKey = requestHandler.getCallKey();
                        File tmpFile = (File) parameters.getResultObject(BdfInstructionConstants.TMPFILE_OBJ);
                        if (tmpFile != null) {
                            return new IllustrationUploadConfirmHtmlProducer(parameters, callKey, tmpFile.getName());
                        } else {
                            return null;
                        }
                    }
                    default:
                        return null;
                }
            }
        }
    }

    private static int getPageType(String page) {
        switch (page) {
            case AlbumDomain.ALBUM_ADVANCEDCOMMANDS_PAGE:
            case AlbumDomain.ALBUM_METADATAFORM_PAGE:
                return ALBUM_ADMIN;
            case AlbumDomain.ALBUM_PAGE:
            case AlbumDomain.ILLUSTRATION_UPLOAD_NEW_PAGE:
            case AlbumDomain.ILLUSTRATION_UPLOAD_REPLACE_PAGE:
            case AlbumDomain.ILLUSTRATION_CHANGE_PAGE:
                return ALBUM_READ;
            case AlbumDomain.ILLUSTRATION_ADMINFORM_PAGE:
            case AlbumDomain.ILLUSTRATION_ADVANCEDCOMMANDS_PAGE:
                return ILLUSTRATION;
            default:
                return 0;
        }
    }

}
