/* BdfServer_Commands - Copyright (c) 2014-2026 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package fr.exemole.bdfserver.commands.importation;

import fr.exemole.bdfserver.api.BdfServer;
import fr.exemole.bdfserver.api.EditSession;
import fr.exemole.bdfserver.api.interaction.Domains;
import fr.exemole.bdfserver.conf.ConfConstants;
import fr.exemole.bdfserver.tools.configuration.ConfigurationUtils;
import fr.exemole.bdfserver.tools.importation.ImportationEngine;
import fr.exemole.bdfserver.tools.instruction.AbstractBdfCommand;
import fr.exemole.bdfserver.tools.instruction.BdfErrors;
import java.io.File;
import java.text.ParseException;
import net.mapeadores.util.exceptions.ShouldNotOccurException;
import net.mapeadores.util.logging.ErrorMessageException;
import net.mapeadores.util.request.RequestMap;
import net.mapeadores.util.text.RelativePath;


/**
 *
 * @author Vincent Calame
 */
public class ImportationEngineCommand extends AbstractBdfCommand {

    public final static String COMMANDNAME = "ImportationEngine";
    public final static String TYPE_PARAMNAME = "type";
    public final static String ROOT_PARAMNAME = "root";
    public final static String PATH_PARAMNAME = "path";
    public final static String TMP_ROOT_PARAMVALUE = "tmp";
    public final static String RUN_ROOT_PARAMVALUE = "run";
    private File source;
    private String engineType;

    public ImportationEngineCommand(BdfServer bdfServer, RequestMap requestMap) {
        super(bdfServer, requestMap);
    }

    @Override
    public boolean needSynchronisation() {
        return true;
    }

    @Override
    protected void doCommand() throws ErrorMessageException {
        try (EditSession session = startEditSession(Domains.IMPORTATION, COMMANDNAME + "/" + engineType)) {
            ImportationEngine.run(session, requestHandler, engineType, source);
        }
        setDone("_ done.importation.engine");
    }

    @Override
    protected void checkParameters() throws ErrorMessageException {
        checkFichothequeAdmin();
        this.engineType = getMandatory(TYPE_PARAMNAME);
        if (!ImportationEngine.isValidEngineType(engineType)) {
            throw BdfErrors.wrongParameterValue(TYPE_PARAMNAME, engineType);
        }
        String root = checkRoot();
        String path = getMandatory(PATH_PARAMNAME).trim();
        RelativePath relativePath;
        try {
            relativePath = RelativePath.parse(path);
        } catch (ParseException pe) {
            throw BdfErrors.wrongParameterValue(PATH_PARAMNAME, path);
        }
        if (relativePath.isEmpty()) {
            throw BdfErrors.emptyMandatoryParameter(PATH_PARAMNAME);
        }
        this.source = getSource(root, relativePath);
    }


    private String checkRoot() throws ErrorMessageException {
        String root = getMandatory(ROOT_PARAMNAME);
        switch (root) {
            case TMP_ROOT_PARAMVALUE:
                return TMP_ROOT_PARAMVALUE;
            case RUN_ROOT_PARAMVALUE:
                return RUN_ROOT_PARAMVALUE;
            default:
                throw BdfErrors.unknownParameterValue(ROOT_PARAMNAME, root);
        }
    }

    private File getSource(String root, RelativePath relativePath) throws ErrorMessageException {
        File rootDir;
        switch (root) {
            case TMP_ROOT_PARAMVALUE:
                rootDir = ConfigurationUtils.getTmpDirectory(bdfServer);
                break;
            case RUN_ROOT_PARAMVALUE:
                rootDir = bdfServer.getBdfServerDirs().getDir(ConfConstants.VAR_RUN);
                break;
            default:
                throw new ShouldNotOccurException("Check done before");
        }
        File src = new File(rootDir, relativePath.toString());
        if (!src.exists()) {
            throw BdfErrors.unknownParameterValue(PATH_PARAMNAME, relativePath.toString());
        }
        return src;
    }

}
