/* BdfServer - Copyright (c) 2020-2026 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package fr.exemole.bdfserver.tools.instruction;

import fr.exemole.bdfserver.api.BdfServerConstants;
import fr.exemole.bdfserver.api.instruction.BdfParameters;
import fr.exemole.bdfserver.api.instruction.PermissionException;
import fr.exemole.bdfserver.api.roles.SphereSupervisor;
import net.fichotheque.Subset;
import net.fichotheque.SubsetItem;
import net.fichotheque.SubsetKey;
import net.fichotheque.corpus.Corpus;
import net.fichotheque.corpus.FicheMeta;
import net.fichotheque.permission.PermissionSummary;


/**
 *
 * @author Vincent Calame
 */
public class PermissionChecker {

    private final BdfParameters bdfParameters;
    private final PermissionSummary permissionSummary;

    public PermissionChecker(BdfParameters bdfParameters) {
        this.bdfParameters = bdfParameters;
        this.permissionSummary = bdfParameters.getPermissionSummary();
    }

    public PermissionChecker checkFichothequeAdmin() throws PermissionException {
        if (!bdfParameters.isFichothequeAdmin()) {
            throw BdfErrors.permission("_ error.permission.admin");
        }
        return this;
    }

    public PermissionChecker checkAdmin(String... actionNames) throws PermissionException {
        if (actionNames != null) {
            for (String actionName : actionNames) {
                if (!permissionSummary.canDo(actionName)) {
                    throw BdfErrors.permission("_ error.permission.admin");
                }
            }
        }
        return this;
    }

    public PermissionChecker checkHistory(Subset subset, int id) throws PermissionException {
        SubsetItem subsetItem = subset.getSubsetItemById(id);
        if ((subsetItem != null) && (subsetItem instanceof FicheMeta)) {
            checkRead((FicheMeta) subsetItem);
        } else {
            checkSubsetAdmin(subset);
        }
        return this;
    }

    public PermissionChecker checkSubsetAdmin(Subset subset) throws PermissionException {
        if (!permissionSummary.isSubsetAdmin(subset.getSubsetKey())) {
            throw BdfErrors.permission("_ error.permission.admin");
        }
        return this;
    }

    public PermissionChecker checkSubsetAccess(Subset subset) throws PermissionException {
        if (!permissionSummary.hasAccess(subset)) {
            throw BdfErrors.permission("_ error.permission.read", subset.getSubsetKeyString());
        }
        return this;
    }

    public PermissionChecker checkWrite(SubsetItem subsetItem) throws PermissionException {
        if (!permissionSummary.canWrite(subsetItem)) {
            throw BdfErrors.permission("_ error.permission.write", subsetItem.getGlobalId());
        }
        return this;
    }

    public PermissionChecker checkRead(SubsetItem subsetItem) throws PermissionException {
        if (!permissionSummary.canRead(subsetItem)) {
            String l10nKey;
            if (subsetItem instanceof FicheMeta) {
                l10nKey = "_ error.permission.readfiche";
            } else {
                l10nKey = "_ error.permission.read";
            }
            throw BdfErrors.permission(l10nKey, subsetItem.getGlobalId());
        }
        return this;
    }

    public PermissionChecker checkFicheCreate(Corpus corpus) throws PermissionException {
        SubsetKey subsetKey = corpus.getSubsetKey();
        if (!permissionSummary.canCreate(subsetKey)) {
            throw BdfErrors.permission("_ error.permission.createfiche", subsetKey.getKeyString());
        }
        return this;
    }

    public PermissionChecker checkSphereSupervisor(String sphereName) throws PermissionException {
        SphereSupervisor sphereSupervisor = (SphereSupervisor) bdfParameters.getBdfServer().getContextObject(BdfServerConstants.SPHERESUPERVISOR_CONTEXTOBJECT);
        if (sphereSupervisor != null) {
            if (!sphereSupervisor.allowCoreChange(bdfParameters.getBdfUser(), sphereName)) {
                throw BdfErrors.permission("_ error.permission.spheresupervisor");
            }
        }
        return this;
    }

    public static PermissionChecker init(BdfParameters bdfParameters) {
        return new PermissionChecker(bdfParameters);
    }

}
