/* BdfServer - Copyright (c) 2020-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package fr.exemole.bdfserver.tools.ficheform.builders.thesaurus;

import fr.exemole.bdfserver.api.ficheform.FicheFormParameters;
import fr.exemole.bdfserver.api.ficheform.include.SubsetItemValue;
import fr.exemole.bdfserver.api.ficheform.include.ThesaurusEntry;
import fr.exemole.bdfserver.api.ui.SubsetIncludeUi;
import fr.exemole.bdfserver.tools.L10nUtils;
import fr.exemole.bdfserver.tools.externalsource.ExternalSourceUtils;
import fr.exemole.bdfserver.tools.ficheform.FicheFormUtils;
import java.util.List;
import net.fichotheque.pointeurs.FichePointeur;
import net.fichotheque.pointeurs.SubsetItemPointeur;
import net.fichotheque.thesaurus.Motcle;
import net.fichotheque.thesaurus.Thesaurus;
import net.mapeadores.util.annotation.Nullable;
import net.mapeadores.util.attr.Attribute;
import net.mapeadores.util.attr.AttributeUtils;
import net.mapeadores.util.attr.Attributes;


/**
 *
 * @author Vincent Calame
 */
public class FicheStyleThesaurusEntryBuilder {

    private final String name;
    private final Thesaurus thesaurus;
    private final MotcleValueListBuilder motcleValueListBuilder = new MotcleValueListBuilder();
    private String label = "";
    private boolean mandatory = false;
    private boolean hasWeightFilter;
    private Attribute idalphaStyle = null;
    private boolean withExternalSource;
    private Attributes attributes = AttributeUtils.EMPTY_ATTRIBUTES;

    public FicheStyleThesaurusEntryBuilder(String name, Thesaurus thesaurus) {
        this.name = name;
        this.thesaurus = thesaurus;
    }

    public FicheStyleThesaurusEntryBuilder setLabel(String label) {
        if (label == null) {
            throw new IllegalArgumentException("label is null");
        }
        this.label = label;
        return this;
    }

    public FicheStyleThesaurusEntryBuilder setMandatory(boolean mandatory) {
        this.mandatory = mandatory;
        return this;
    }

    public FicheStyleThesaurusEntryBuilder setAttributes(Attributes attributes) {
        if (attributes == null) {
            this.attributes = AttributeUtils.EMPTY_ATTRIBUTES;
        } else {
            this.attributes = attributes;
        }
        return this;
    }

    public FicheStyleThesaurusEntryBuilder setIdalphaStyle(Attribute idalphaStyle) {
        this.idalphaStyle = idalphaStyle;
        return this;
    }

    public FicheStyleThesaurusEntryBuilder setHasWeightFilter(boolean hasWeightFilter) {
        this.hasWeightFilter = hasWeightFilter;
        return this;
    }

    public FicheStyleThesaurusEntryBuilder addMotcle(Motcle motcle, int weight) {
        motcleValueListBuilder.add(motcle, weight);
        return this;
    }

    public FicheStyleThesaurusEntryBuilder setWithExternalSource(boolean withExternalSource) {
        this.withExternalSource = withExternalSource;
        return this;
    }

    public ThesaurusEntry.FicheStyle toFicheStyleThesaurusEntry() {
        return new InternalFicheStyleThesaurusEntry(name, label, mandatory, attributes, thesaurus, idalphaStyle, hasWeightFilter, motcleValueListBuilder.toList(), withExternalSource);
    }

    @Nullable
    public static FicheStyleThesaurusEntryBuilder check(FichePointeur fichePointeur, FicheFormParameters ficheFormParameters, SubsetIncludeUi includeUi) {
        Thesaurus thesaurus = (Thesaurus) ficheFormParameters.getBdfServer().getFichotheque().getSubset(includeUi.getSubsetKey());
        if (thesaurus == null) {
            return null;
        }
        boolean isMandatory = includeUi.isMandatory();
        boolean withExternalSource = (ExternalSourceUtils.getExternalSource(ficheFormParameters.getBdfServer(), thesaurus) != null);
        FicheStyleThesaurusEntryBuilder builder = init(includeUi.getName(), thesaurus)
                .setMandatory(isMandatory)
                .setAttributes(includeUi.getAttributes())
                .setIdalphaStyle(FicheFormUtils.getIdalphaStyle(includeUi, thesaurus))
                .setLabel(L10nUtils.toLabelString(ficheFormParameters, includeUi))
                .setHasWeightFilter(includeUi.getExtendedIncludeKey().hasWeightFilter())
                .setWithExternalSource(withExternalSource);
        SubsetItemPointeur pointeur = FicheFormUtils.checkMasterPointeur(fichePointeur, ficheFormParameters, includeUi);
        if (!pointeur.isEmpty()) {
            builder.motcleValueListBuilder.populate(pointeur.getJunctions(thesaurus), includeUi, true);
        }
        if ((builder.motcleValueListBuilder.isEmpty()) && (includeUi.isObsolete())) {
            return null;
        }
        return builder;
    }

    public static FicheStyleThesaurusEntryBuilder init(String name, Thesaurus thesaurus) {
        return new FicheStyleThesaurusEntryBuilder(name, thesaurus);
    }


    private static class InternalFicheStyleThesaurusEntry implements ThesaurusEntry.FicheStyle {

        private final String name;
        private final String label;
        private final boolean mandatory;
        private final Attributes attributes;
        private final Thesaurus thesaurus;
        private final Attribute idalphaStyle;
        private final boolean withWeightFilter;
        private final List<SubsetItemValue> list;
        private final boolean withExternalSource;

        private InternalFicheStyleThesaurusEntry(String name, String label, boolean mandatory, Attributes attributes, Thesaurus thesaurus, Attribute idalphaStyle, boolean withWeightFilter, List<SubsetItemValue> list, boolean withExternalSource) {
            this.name = name;
            this.label = label;
            this.mandatory = mandatory;
            this.attributes = attributes;
            this.thesaurus = thesaurus;
            this.idalphaStyle = idalphaStyle;
            this.withWeightFilter = withWeightFilter;
            this.list = list;
            this.withExternalSource = withExternalSource;
        }

        @Override
        public String getIncludeName() {
            return name;
        }

        @Override
        public String getLabel() {
            return label;
        }

        @Override
        public boolean isMandatory() {
            return mandatory;
        }

        @Override
        public Attributes getAttributes() {
            return attributes;
        }

        @Override
        public Thesaurus getThesaurus() {
            return thesaurus;
        }

        @Override
        public Attribute getIdalphaStyle() {
            return idalphaStyle;
        }

        @Override
        public boolean hasWeightFilter() {
            return withWeightFilter;
        }

        @Override
        public List<SubsetItemValue> getValueList() {
            return list;
        }

        @Override
        public boolean isWithExternalSource() {
            return withExternalSource;
        }

    }

}
