/* BdfServer - Copyright (c) 2013-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package fr.exemole.bdfserver.tools.importation;

import fr.exemole.bdfserver.api.BdfServer;
import fr.exemole.bdfserver.tools.configuration.ConfigurationUtils;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import net.fichotheque.importation.CorpusImport;
import net.fichotheque.importation.LabelImport;
import net.fichotheque.importation.ThesaurusImport;
import net.fichotheque.xml.importation.CorpusImportXMLPart;
import net.fichotheque.xml.importation.LabelImportXMLPart;
import net.fichotheque.xml.importation.ThesaurusImportXMLPart;
import net.mapeadores.util.xml.AppendableXMLWriter;
import net.mapeadores.util.xml.XMLUtils;


/**
 *
 * @author Vincent Calame
 */
public final class ImportationFileUtils {

    private final static DateTimeFormatter NAME_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd'_'HH-mm-ss");

    private ImportationFileUtils() {
    }

    public static File saveTmpXml(BdfServer bdfServer, LabelImport labelImport) throws IOException {
        File xmlTmpFile = createXmlTmpFile(bdfServer, ImportationEngine.LABEL_IMPORT);
        try (Writer writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(xmlTmpFile), "UTF-8"))) {
            AppendableXMLWriter xmlWriter = XMLUtils.toXMLWriter(writer);
            xmlWriter.appendXMLDeclaration();
            LabelImportXMLPart labelImportXMLPart = new LabelImportXMLPart(xmlWriter);
            labelImportXMLPart.addLabelImport(labelImport);
        }
        return xmlTmpFile;
    }

    public static File saveTmpXml(BdfServer bdfServer, ThesaurusImport thesaurusImport) throws IOException {
        File xmlTmpFile = createXmlTmpFile(bdfServer, ImportationEngine.THESAURUS_IMPORT);
        try (Writer writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(xmlTmpFile), "UTF-8"))) {
            AppendableXMLWriter xmlWriter = XMLUtils.toXMLWriter(writer);
            xmlWriter.appendXMLDeclaration();
            ThesaurusImportXMLPart thesaurusImportXMLPart = new ThesaurusImportXMLPart(xmlWriter);
            thesaurusImportXMLPart.addThesaurusImport(thesaurusImport);
        }
        return xmlTmpFile;
    }

    public static File saveTmpXml(BdfServer bdfServer, CorpusImport corpusImport) throws IOException {
        File xmlTmpFile = createXmlTmpFile(bdfServer, ImportationEngine.CORPUS_IMPORT);
        try (Writer writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(xmlTmpFile), "UTF-8"))) {
            AppendableXMLWriter xmlWriter = XMLUtils.toXMLWriter(writer);
            xmlWriter.appendXMLDeclaration();
            CorpusImportXMLPart corpusImportXMLPart = new CorpusImportXMLPart(xmlWriter);
            corpusImportXMLPart.addCorpusImport(corpusImport);
        }
        return xmlTmpFile;
    }

    public static File createXmlTmpFile(BdfServer bdfServer, String basename) {
        synchronized (bdfServer) {
            File tmpDirectory = ConfigurationUtils.getTmpDirectory(bdfServer, true);
            String name = getTmpName(tmpDirectory, basename, ".xml");
            return new File(tmpDirectory, name);
        }
    }

    public static File createTmpDir(BdfServer bdfServer, String basename) throws IOException {
        synchronized (bdfServer) {
            File tmpDirectory = ConfigurationUtils.getTmpDirectory(bdfServer, true);
            String name = getTmpName(tmpDirectory, basename, "");
            File dir = new File(tmpDirectory, name);
            dir.mkdir();
            return dir;
        }
    }

    private static String getTmpName(File dir, String basename, String extension) {
        String prefix = basename + "_" + NAME_FORMAT.format(LocalDateTime.now());
        String name = prefix + extension;
        File file = new File(dir, name);
        if (!file.exists()) {
            return name;
        }
        int p = 2;
        while (true) {
            name = prefix + "_" + p + extension;
            file = new File(name);
            if (!file.exists()) {
                return name;
            } else {
                p++;
            }
        }
    }

}
