/* OdLib_Io - Copyright (c) 2013-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.mapeadores.opendocument.io.odtable;

import net.mapeadores.util.annotation.Nullable;
import net.mapeadores.util.money.ExtendedCurrency;


/**
 *
 * @author Vincent Calame
 */
public final class StyleKey {

    private final short styleFamily;
    private final String styleName;
    private final ExtendedCurrency currency;
    private final String pattern;

    private StyleKey(String parentStyleName) {
        this.styleFamily = OdColumnDef.STANDARD_STYLE_FAMILY;
        this.styleName = parentStyleName;
        this.currency = null;
        this.pattern = null;
    }

    private StyleKey(ExtendedCurrency currency, String parentStyleName) {
        this.styleFamily = OdColumnDef.CURRENCY_STYLE_FAMILY;
        this.styleName = parentStyleName;
        this.currency = currency;
        this.pattern = null;
    }

    private StyleKey(String pattern, String parentStyleName) {
        this.styleFamily = OdColumnDef.DATE_STYLE_FAMILY;
        this.styleName = parentStyleName;
        this.currency = null;
        this.pattern = pattern;
    }

    public short getStyleFamliy() {
        return styleFamily;
    }

    public boolean isStandardStyle() {
        return (styleFamily == OdColumnDef.STANDARD_STYLE_FAMILY);
    }

    public boolean isDateStyle() {
        return (styleFamily == OdColumnDef.DATE_STYLE_FAMILY);
    }

    public boolean isCurrencyStyle() {
        return (styleFamily == OdColumnDef.CURRENCY_STYLE_FAMILY);
    }

    @Nullable
    public ExtendedCurrency getCurrency() {
        return currency;
    }

    @Nullable
    public String getPattern() {
        return pattern;
    }

    @Nullable
    public String getParentStyleName() {
        return styleName;
    }

    @Override
    public int hashCode() {
        int code = styleFamily;
        if (currency != null) {
            code += currency.hashCode();
        }
        if (styleName != null) {
            code += styleName.hashCode();
        }
        if (pattern != null) {
            code += pattern.hashCode();
        }
        return code;
    }

    @Override
    public boolean equals(Object other) {
        if (other == null) {
            return false;
        }
        if (other == this) {
            return true;
        }
        if (other.getClass() != this.getClass()) {
            return false;
        }
        StyleKey otherStyleKey = (StyleKey) other;
        if (otherStyleKey.styleFamily != this.styleFamily) {
            return false;
        }
        if (otherStyleKey.currency == null) {
            if (this.currency != null) {
                return false;
            }
        } else if (this.currency == null) {
            return false;
        } else if (!otherStyleKey.currency.equals(this.currency)) {
            return false;
        }
        if (otherStyleKey.pattern == null) {
            if (this.pattern != null) {
                return false;
            }
        } else if (this.pattern == null) {
            return false;
        } else if (!otherStyleKey.pattern.equals(this.pattern)) {
            return false;
        }
        if (otherStyleKey.styleName == null) {
            if (this.styleName == null) {
                return true;
            } else {
                return false;
            }
        } else if (this.styleName == null) {
            return false;
        } else {
            return otherStyleKey.styleName.equals(this.styleName);
        }
    }

    public static StyleKey newStandardInstance(String parentStyleName) {
        return new StyleKey(parentStyleName);
    }

    public static StyleKey newIsoDateInstance(String parentStyleName) {
        return new StyleKey("y-MM-dd", parentStyleName);
    }

    public static StyleKey newDateInstance(String pattern) {
        return new StyleKey(pattern, null);
    }

    public static StyleKey newDateInstance(String pattern, String parentStyleName) {
        return new StyleKey(pattern, parentStyleName);
    }

    public static StyleKey newCurrencyInstance(ExtendedCurrency currency) {
        return newCurrencyInstance(currency, null);
    }

    public static StyleKey newCurrencyInstance(ExtendedCurrency currency, String parentStyleName) {
        return new StyleKey(currency, parentStyleName);
    }

}
