/* FichothequeLib_Tools - Copyright (c) 2007-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.tools.format.parsers;

import net.fichotheque.corpus.FicheMeta;
import net.fichotheque.format.FormatContext;
import net.fichotheque.format.FormatSource;
import net.fichotheque.format.formatters.FicheMetaFormatter;
import net.fichotheque.tools.format.DefaultPatternCatalog;
import net.fichotheque.tools.format.JsonParameters;
import net.fichotheque.tools.format.catalogs.FicheMetaFormatterCatalog;
import net.fichotheque.utils.FormatterUtils;
import net.mapeadores.util.format.FormatConstants;
import net.mapeadores.util.instruction.Argument;
import net.mapeadores.util.instruction.Instruction;
import net.mapeadores.util.instruction.InstructionResolver;
import net.mapeadores.util.instruction.InstructionResolverProvider;
import net.mapeadores.util.logging.ErrorMessageException;
import net.mapeadores.util.logging.MessageHandler;


/**
 *
 * @author Vincent Calame
 */
public final class FicheMetaFormatterParser {

    private final static JsonParameters DEFAULT_JSONPARAMETERS = JsonParameters.build("type", "id");

    private FicheMetaFormatterParser() {
    }

    public static FicheMetaFormatter parse(String pattern, FormatContext formatContext, boolean withWeightFilter, boolean withCorpusName, MessageHandler messageHandler) {
        if (pattern == null) {
            pattern = DefaultPatternCatalog.FICHEMETA;
        }
        try {
            return parse(pattern, formatContext, withWeightFilter, withCorpusName);
        } catch (ErrorMessageException fe) {
            messageHandler.addMessage(FormatConstants.SEVERE_PATTERN, fe.getErrorMessage());
            return null;
        }
    }

    public static FicheMetaFormatter parse(String pattern, FormatContext formatContext, boolean withWeightFilter, boolean withCorpusName) throws ErrorMessageException {
        Object[] partArray = FormatterUtils.parsePattern(new InternalInstructionResolver(formatContext, withWeightFilter, withCorpusName), pattern);
        return new InternalFicheMetaFormatter(partArray);
    }


    private static class InternalFicheMetaFormatter implements FicheMetaFormatter {

        private final Object[] partArray;

        private InternalFicheMetaFormatter(Object[] partArray) {
            this.partArray = partArray;
        }

        @Override
        public String formatFicheMeta(FicheMeta ficheMeta, int weight, FormatSource formatSource) {
            StringBuilder buf = new StringBuilder();
            int length = partArray.length;
            for (int i = 0; i < length; i++) {
                Object obj = partArray[i];
                if (obj instanceof String) {
                    buf.append((String) obj);
                    continue;
                }
                FicheMetaFormatter formatter = (FicheMetaFormatter) obj;
                buf.append(formatter.formatFicheMeta(ficheMeta, weight, formatSource));
            }
            return buf.toString();
        }

    }


    private static class InternalInstructionResolver implements InstructionResolver {

        private final InstructionResolverProvider provider;
        private final boolean withWeightFilter;
        private final boolean withCorpusName;


        private InternalInstructionResolver(FormatContext formatContext, boolean withWeightFilter, boolean withCorpusName) {
            this.provider = formatContext.getInstructionResolverProvider();
            this.withWeightFilter = withWeightFilter;
            this.withCorpusName = withCorpusName;
        }

        @Override
        public Object resolve(Instruction instruction) throws ErrorMessageException {
            InstructionResolver resolver = provider.getInstructionResolver(FicheMetaFormatter.class);
            if (resolver != null) {
                Object formatter = resolver.resolve(instruction);
                if (formatter != null) {
                    return formatter;
                }
            }
            Argument arg1 = instruction.get(0);
            String key = arg1.getKey();
            switch (key) {
                case "corpus":
                    return FicheMetaFormatterCatalog.CORPUS;
                case "formsyntax":
                    return new FicheMetaFormatterCatalog.FormSyntax(withWeightFilter, withCorpusName);
                case "id":
                case "idcorpus":
                    return FicheMetaFormatterCatalog.ID;
                case "json": {
                    JsonParameters jsonParameters = JsonParameters.fromInstruction(instruction, DEFAULT_JSONPARAMETERS);
                    return new FicheMetaFormatterCatalog.Json(jsonParameters);
                }
                case "lang":
                    return FicheMetaFormatterCatalog.LANG;
                case "weight":
                case "poids":
                    return FicheMetaFormatterCatalog.WEIGHT;
                case "title":
                case "titre":
                    return FicheMetaFormatterCatalog.TITLE;
                default:
                    return null;
            }
        }

    }


}
