/* FichothequeLib_Tools - Copyright (c) 2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.tools.format.catalogs;

import net.fichotheque.tools.format.JsonParameters;
import java.io.IOException;
import java.util.List;
import net.fichotheque.addenda.Document;
import net.fichotheque.addenda.Version;
import net.fichotheque.format.FormatSource;
import net.fichotheque.format.formatters.DocumentFormatter;
import net.fichotheque.json.DocumentJson;
import net.mapeadores.util.exceptions.ShouldNotOccurException;
import net.mapeadores.util.json.JSONWriter;
import net.mapeadores.util.localisation.LangContext;


/**
 *
 * @author Vincent Calame
 */
public final class DocumentFormatterCatalog {

    public final static DocumentFormatter ADDENDA = (document, weight, formatSource) -> {
        return document.getSubsetName();
    };
    public final static DocumentFormatter BASENAME = (document, weight, formatSource) -> {
        return document.getBasename();
    };
    public final static DocumentFormatter EXTENSIONS = (document, weight, formatSource) -> {
        List<Version> versionList = document.getVersionList();
        if (versionList.size() == 1) {
            return '.' + versionList.get(0).getExtension();
        } else {
            StringBuilder buf = new StringBuilder();
            for (Version version : versionList) {
                buf.append('[');
                buf.append('.');
                buf.append(version.getExtension());
                buf.append(']');
            }
            return buf.toString();
        }
    };
    public final static DocumentFormatter ID = (document, weight, formatSource) -> {
        return String.valueOf(document.getId());
    };
    public final static DocumentFormatter WEIGHT = (document, weight, formatSource) -> {
        return String.valueOf(weight);
    };
    public final static DocumentFormatter WITHWEIGHTFILTER_FORMSYNTAX = new InternalFormSyntax(true);
    public final static DocumentFormatter WITHOUTWEIGHTFILTER_FORMSYNTAX = new InternalFormSyntax(false);


    private DocumentFormatterCatalog() {

    }


    public static class Json implements DocumentFormatter {

        private final JsonParameters jsonParameters;
        private final LangContext customLangContext;

        public Json(JsonParameters jsonParameters, LangContext customLangContext) {
            this.jsonParameters = jsonParameters;
            this.customLangContext = customLangContext;
        }

        @Override
        public String formatDocument(Document document, int weight, FormatSource formatSource) {
            LangContext langContext;
            if (customLangContext != null) {
                langContext = customLangContext;
            } else {
                langContext = formatSource.getLangContext();
            }
            StringBuilder buf = new StringBuilder();
            JSONWriter jw = new JSONWriter(buf);
            try {
                jw.object();
                DocumentJson.properties(jw, document, langContext, jsonParameters.getPropertyEligibility());
                jw.endObject();
            } catch (IOException ioe) {
                throw new ShouldNotOccurException(ioe);
            }
            return buf.toString();
        }

    }


    private static class InternalFormSyntax implements DocumentFormatter {

        private final boolean withWeightFilter;

        private InternalFormSyntax(boolean withWeightFilter) {
            this.withWeightFilter = withWeightFilter;
        }

        @Override
        public String formatDocument(Document document, int weight, FormatSource formatSource) {
            if ((withWeightFilter) || (weight == 1)) {
                return String.valueOf(document.getId());
            } else {
                StringBuilder buf = new StringBuilder();
                buf.append(String.valueOf(document.getId()));
                buf.append(" <");
                buf.append(weight);
                buf.append(">");
                return buf.toString();
            }
        }

    }

}
