/* FichothequeLib_API - Copyright (c) 2015-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.json;

import java.io.IOException;
import java.text.NumberFormat;
import java.util.Locale;
import net.fichotheque.addenda.Document;
import net.fichotheque.addenda.Version;
import net.mapeadores.util.json.JSONWriter;
import net.mapeadores.util.json.PropertyEligibility;
import net.mapeadores.util.localisation.Lang;
import net.mapeadores.util.localisation.LangContext;
import net.mapeadores.util.localisation.ListLangContext;
import net.mapeadores.util.localisation.LocalisationUtils;
import net.mapeadores.util.localisation.UserLangContext;
import net.mapeadores.util.primitives.FileLength;


/**
 *
 * @author Vincent Calame
 */
public final class DocumentJson {

    private DocumentJson() {

    }

    public static void properties(JSONWriter jw, Document document, LangContext langContext, PropertyEligibility propertyEligibility) throws IOException {
        if (propertyEligibility.includeProperty("type")) {
            jw.key("type")
                    .value("document");
        }
        if (propertyEligibility.includeProperty("addenda")) {
            jw.key("addenda")
                    .value(document.getSubsetName());
        }
        if (propertyEligibility.includeProperty("id")) {
            jw.key("id")
                    .value(document.getId());
        }
        if (propertyEligibility.includeProperty("basename")) {
            jw.key("basename")
                    .value(document.getBasename());
        }
        if (propertyEligibility.includeProperty("versions")) {
            jw.key("versions");
            jw.array();
            for (Version version : document.getVersionList()) {
                jw.object();
                jw.key("extension")
                        .value(version.getExtension());
                if (propertyEligibility.includeProperty("size")) {
                    FileLength fileLength = version.getFileLength();
                    float value = fileLength.getRoundedValue();
                    boolean isKibiOctet = (fileLength.getRoundType() == FileLength.KB_ROUND);
                    jw.key("size");
                    jw.object();
                    {
                        jw.key("value")
                                .value(value);
                        jw.key("unit")
                                .value((isKibiOctet) ? "Ki" : "Mi");
                        jw.key("labels");
                        jw.object();
                        {
                            if (langContext instanceof ListLangContext) {
                                for (ListLangContext.Unit unit : (ListLangContext) langContext) {
                                    addSizeLabel(jw, unit.getLang(), unit.getFormatLocale(), value, isKibiOctet);
                                }
                            } else if (langContext instanceof UserLangContext) {
                                UserLangContext userLangContext = (UserLangContext) langContext;
                                addSizeLabel(jw, userLangContext.getWorkingLang(), userLangContext.getFormatLocale(), value, isKibiOctet);
                            }
                        }
                        jw.endObject();
                    }
                    jw.endObject();
                }
                jw.endObject();
            }
            jw.endArray();
        }
    }

    private static void addSizeLabel(JSONWriter jw, Lang lang, Locale formatLocale, float value, boolean isKibiOctet) throws IOException {
        NumberFormat format = NumberFormat.getInstance(formatLocale);
        StringBuilder buf = new StringBuilder();
        buf.append(format.format(value));
        buf.append(' ');
        if (isKibiOctet) {
            buf.append(LocalisationUtils.getKibiOctet(lang));
        } else {
            buf.append(LocalisationUtils.getMebiOctet(lang));
        }
        jw.key(lang.toString())
                .value(buf.toString());
    }

}
