/* BdfServer - Copyright (c) 2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package fr.exemole.bdfserver.tools.ficheform.builders.field;

import fr.exemole.bdfserver.api.BdfServerConstants;
import fr.exemole.bdfserver.api.ficheform.FicheFormParameters;
import fr.exemole.bdfserver.api.ui.FieldUi;
import net.fichotheque.corpus.metadata.CorpusField;
import net.fichotheque.corpus.metadata.FieldKey;
import net.fichotheque.pointeurs.FichePointeur;
import net.fichotheque.utils.CorpusMetadataUtils;
import net.mapeadores.util.annotation.Nullable;
import fr.exemole.bdfserver.api.ficheform.FormEntry;


/**
 *
 * @author Vincent Calame
 */
public final class FieldEntryFactory {

    private FieldEntryFactory() {

    }

    @Nullable
    public static FormEntry.Field newInstance(FichePointeur fichePointeur, FicheFormParameters ficheFormParameters, CorpusField corpusField, FieldUi fieldUi) {
        switch (corpusField.getFieldString()) {
            case FieldKey.SPECIAL_ID:
                return null;
            case FieldKey.SPECIAL_LANG:
                return LangFieldEntryBuilder.build(fichePointeur, ficheFormParameters, corpusField, fieldUi)
                        .toLangFieldEntry();
        }
        if ((corpusField.isProp()) && (corpusField.isSubfieldDisplay())) {
            switch (corpusField.getFicheItemType()) {
                case CorpusField.PERSON_FIELD: {
                    PersonnePropEntryBuilder builder = PersonnePropEntryBuilder.check(fichePointeur, ficheFormParameters, corpusField, fieldUi);
                    if (builder != null) {
                        return builder.toPersonnePropEntry();
                    } else {
                        return null;
                    }
                }
                case CorpusField.GEOPOINT_FIELD: {
                    GeopointPropEntryBuilder builder = GeopointPropEntryBuilder.check(fichePointeur, ficheFormParameters, corpusField, fieldUi);
                    if (builder != null) {
                        return builder.toGeopointPropEntry();
                    } else {
                        return null;
                    }
                }
                case CorpusField.IMAGE_FIELD: {
                    ImagePropEntryBuilder builder = ImagePropEntryBuilder.check(fichePointeur, ficheFormParameters, corpusField, fieldUi);
                    if (builder != null) {
                        return builder.toImagePropEntry();
                    } else {
                        return null;
                    }
                }
            }
        }
        if (CorpusMetadataUtils.isCurrenciesProp(corpusField)) {
            AmountPropEntryBuilder builder = AmountPropEntryBuilder.check(fichePointeur, ficheFormParameters, corpusField, fieldUi);
            if (builder != null) {
                return builder.toAmountPropEntry();
            } else {
                return null;
            }
        }
        if (CorpusMetadataUtils.isCurrenciesInfo(corpusField)) {
            AmountInfoEntryBuilder builder = AmountInfoEntryBuilder.check(fichePointeur, ficheFormParameters, corpusField, fieldUi);
            if (builder != null) {
                return builder.toAmountInfoEntry();
            } else {
                return null;
            }
        }
        return newSimpleFieldInstance(fichePointeur, ficheFormParameters, corpusField, fieldUi, ficheFormParameters.getCustomDefaultValue(fieldUi.getName()));
    }


    @Nullable
    public static FormEntry.Field newSimpleFieldInstance(FichePointeur fichePointeur, FicheFormParameters ficheFormParameters, CorpusField corpusField, FieldUi fieldUi, @Nullable String defaultValue) {
        if (corpusField.isSection()) {
            TextFieldEntryBuilder builder = TextFieldEntryBuilder.check(fichePointeur, ficheFormParameters, corpusField, fieldUi, defaultValue);
            if (builder != null) {
                return builder.toTextFieldEntry();
            } else {
                return null;
            }
        }
        switch (fieldUi.getOptionValue(BdfServerConstants.INPUTTYPE_OPTION)) {
            case BdfServerConstants.INPUT_HIDDEN: {
                HiddenFieldEntryBuilder builder = HiddenFieldEntryBuilder.check(fichePointeur, ficheFormParameters, corpusField, fieldUi, defaultValue);
                if (builder != null) {
                    return builder.toHiddenFieldEntry();
                } else {
                    return null;
                }
            }
            default: {
                ItemFieldEntryBuilder builder = ItemFieldEntryBuilder.check(fichePointeur, ficheFormParameters, corpusField, fieldUi, defaultValue);
                if (builder != null) {
                    return builder.toItemFieldEntry();
                } else {
                    return null;
                }
            }
        }
    }

}
