/* FichothequeLib_Xml - Copyright (c) 2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.xml.extraction;

import java.time.format.FormatStyle;
import net.fichotheque.corpus.metadata.CorpusField;
import net.fichotheque.corpus.metadata.FieldOptionConstants;
import net.fichotheque.extraction.ExtractionConstants;
import net.fichotheque.extraction.FilterParameters;
import net.mapeadores.util.date.FuzzyDate;
import net.mapeadores.util.format.FormatConstants;


/**
 *
 * @author Vincent Calame
 */
public abstract class ExtractOptions {

    public ExtractOptions() {

    }

    public static ExtractOptions build(CorpusField corpusField, FilterParameters filterParameters) {
        switch (corpusField.getFicheItemType()) {
            case CorpusField.IMAGE_FIELD:
            case CorpusField.LINK_FIELD:
                return new Ref(corpusField.getStringOption(FieldOptionConstants.BASEURL_OPTION));
            case CorpusField.DATE_FIELD: {
                String truncateDateType = filterParameters.getFirstValue(ExtractionConstants.DATELABELTRUNCATE_PARAM);
                if (truncateDateType != null) {
                    try {
                        truncateDateType = FuzzyDate.checkDateType(truncateDateType);
                    } catch (IllegalArgumentException iae) {
                    }
                }
                FormatStyle formatStyle = null;
                String dateStyle = filterParameters.getFirstValue(ExtractionConstants.DATESTYLE_PARAM);
                if (dateStyle != null) {
                    formatStyle = FormatConstants.getMatchingFormatStyle(dateStyle);
                }
                if ((truncateDateType != null) || (formatStyle != null)) {
                    return new Date(truncateDateType, formatStyle);
                } else {
                    return null;
                }
            }
            case CorpusField.AMOUNT_FIELD: {
                String paramValue = filterParameters.getFirstValue(ExtractionConstants.AMOUNTSUBUNIT_PARAM);
                if ((paramValue != null) && (paramValue.equals(FormatConstants.FORCE_AMOUNTSUBUNIT))) {
                    return new Amount(true);
                }

            }
            default:
                return null;
        }
    }


    public static class Date extends ExtractOptions {

        private final String labelType;
        private final FormatStyle formatStyle;

        public Date(String labelType, FormatStyle formatStyle) {
            this.labelType = labelType;
            this.formatStyle = formatStyle;
        }

        public String getTruncateDateType() {
            return labelType;
        }

        public FormatStyle getStyle() {
            return formatStyle;
        }

    }


    public static class Ref extends ExtractOptions {

        private final String baseUrl;

        public Ref(String baseUrl) {
            this.baseUrl = baseUrl;
        }

        public String getBaseUrl() {
            return baseUrl;
        }

    }


    public static class Amount extends ExtractOptions {

        private final boolean forceSubunit;

        public Amount(boolean forceSubunit) {
            this.forceSubunit = forceSubunit;
        }

        public boolean isForceSubunit() {
            return forceSubunit;
        }

    }

}
