/* FichothequeLib_Tools - Copyright (c) 2012-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.tools.format.parsers;

import net.fichotheque.album.Illustration;
import net.fichotheque.format.FormatContext;
import net.fichotheque.format.FormatSource;
import net.fichotheque.format.formatters.IllustrationFormatter;
import net.fichotheque.tools.format.JsonParameters;
import net.fichotheque.tools.format.catalogs.IllustrationFormatterCatalog;
import net.fichotheque.tools.format.patterndefs.DefaultPattern;
import net.fichotheque.utils.FormatterUtils;
import net.mapeadores.util.format.FormatConstants;
import net.mapeadores.util.instruction.Argument;
import net.mapeadores.util.instruction.Instruction;
import net.mapeadores.util.instruction.InstructionResolver;
import net.mapeadores.util.instruction.InstructionResolverProvider;
import net.mapeadores.util.logging.ErrorMessageException;
import net.mapeadores.util.logging.MessageHandler;


/**
 *
 * @author Vincent Calame
 */
public final class IllustrationFormatterParser {

    private final static JsonParameters DEFAULT_JSONPARAMETERS = JsonParameters.build("type", "id", "format");

    private IllustrationFormatterParser() {
    }

    public static IllustrationFormatter parse(String pattern, FormatContext formatContext, boolean withWeightFilter, MessageHandler messageHandler) {
        if (pattern == null) {
            pattern = DefaultPattern.illustration();
        }
        try {
            return parse(pattern, formatContext, withWeightFilter);
        } catch (ErrorMessageException fe) {
            messageHandler.addMessage(FormatConstants.SEVERE_PATTERN, fe.getErrorMessage());
            return null;
        }
    }

    public static IllustrationFormatter parse(String pattern, FormatContext formatContext, boolean withWeightFilter) throws ErrorMessageException {
        Object[] partArray = FormatterUtils.parsePattern(new InternalInstructionResolver(formatContext, withWeightFilter), pattern);
        return new InternalIllustrationFormatter(partArray);
    }


    private static class InternalIllustrationFormatter implements IllustrationFormatter {

        private final Object[] partArray;

        private InternalIllustrationFormatter(Object[] partArray) {
            this.partArray = partArray;
        }

        @Override
        public String formatIllustration(Illustration illustration, int weight, FormatSource formatSource) {
            StringBuilder buf = new StringBuilder();
            int length = partArray.length;
            for (int i = 0; i < length; i++) {
                Object obj = partArray[i];
                if (obj instanceof String) {
                    buf.append((String) obj);
                    continue;
                }
                IllustrationFormatter formatter = (IllustrationFormatter) obj;
                buf.append(formatter.formatIllustration(illustration, weight, formatSource));
            }
            return buf.toString();
        }

    }


    private static class InternalInstructionResolver implements InstructionResolver {

        private final InstructionResolverProvider provider;
        private final boolean withWeightFilter;

        private InternalInstructionResolver(FormatContext formatContext, boolean withWeightFilter) {
            this.provider = formatContext.getInstructionResolverProvider();
            this.withWeightFilter = withWeightFilter;
        }

        @Override
        public Object resolve(Instruction instruction) throws ErrorMessageException {
            InstructionResolver resolver = provider.getInstructionResolver(IllustrationFormatter.class);
            if (resolver != null) {
                Object formatter = resolver.resolve(instruction);
                if (formatter != null) {
                    return formatter;
                }
            }
            Argument arg1 = instruction.get(0);
            String key = arg1.getKey();
            switch (key) {
                case "album":
                    return IllustrationFormatterCatalog.ALBUM;
                case "format":
                    return IllustrationFormatterCatalog.FORMAT;
                case "formsyntax":
                    if (withWeightFilter) {
                        return IllustrationFormatterCatalog.WITHWEIGHTFILTER_FORMSYNTAX;
                    } else {
                        return IllustrationFormatterCatalog.WITHOUTWEIGHTFILTER_FORMSYNTAX;
                    }
                case "id":
                case "idalbum":
                    return IllustrationFormatterCatalog.ID;
                case "json": {
                    JsonParameters jsonParameters = JsonParameters.fromInstruction(instruction, DEFAULT_JSONPARAMETERS);
                    return new IllustrationFormatterCatalog.Json(jsonParameters);
                }
                case "weight":
                case "poids":
                    return IllustrationFormatterCatalog.WEIGHT;
                default:
                    return null;
            }
        }

    }


}
