/* FichothequeLib_Tools - Copyright (c) 2022-2025 Vincent Calame - Exemole
 * Logiciel libre donné sous triple licence :
 * 1) selon les termes de la CeCILL V2
 * 2) selon les termes de l’EUPL V.1.1
 * 3) selon les termes de la GNU GPLv3
 * Voir le fichier licences.txt
 */


package net.fichotheque.tools.corpus;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Predicate;
import net.fichotheque.corpus.fiche.Fiche;
import net.fichotheque.corpus.fiche.Info;
import net.fichotheque.corpus.fiche.Prop;
import net.fichotheque.corpus.fiche.Section;
import net.fichotheque.corpus.metadata.FieldKey;


/**
 *
 * @author Vincent Calame
 */
public final class PurgeEngine {

    private final Predicate<FieldKey> fieldPredicate;
    private final boolean subtitlePurge;

    public PurgeEngine(Predicate<FieldKey> fieldPredicate) {
        this.fieldPredicate = fieldPredicate;
        this.subtitlePurge = !fieldPredicate.test(FieldKey.SUBTITLE);
    }

    public void purge(Fiche fiche) {
        if (subtitlePurge) {
            fiche.setSubtitle(null);
        }
        purgePropList(fiche);
        purgeInfoList(fiche);
        purgeSectionList(fiche);
    }

    private void purgePropList(Fiche fiche) {
        List<FieldKey> purgeList = new ArrayList<FieldKey>();
        for (Prop prop : fiche.getPropList()) {
            FieldKey fieldKey = prop.getFieldKey();
            if (!fieldPredicate.test(fieldKey)) {
                purgeList.add(fieldKey);
            }
        }
        if (!purgeList.isEmpty()) {
            for (FieldKey fieldKey : purgeList) {
                fiche.setProp(fieldKey, null);
            }
        }
    }

    private void purgeInfoList(Fiche fiche) {
        List<FieldKey> purgeList = new ArrayList<FieldKey>();
        for (Info info : fiche.getInfoList()) {
            FieldKey fieldKey = info.getFieldKey();
            if (!fieldPredicate.test(fieldKey)) {
                purgeList.add(fieldKey);
            }
        }
        if (!purgeList.isEmpty()) {
            for (FieldKey fieldKey : purgeList) {
                fiche.setInfo(fieldKey, null);
            }
        }
    }

    private void purgeSectionList(Fiche fiche) {
        List<FieldKey> purgeList = new ArrayList<FieldKey>();
        for (Section section : fiche.getSectionList()) {
            FieldKey fieldKey = section.getFieldKey();
            if (!fieldPredicate.test(fieldKey)) {
                purgeList.add(fieldKey);
            }
        }
        if (!purgeList.isEmpty()) {
            for (FieldKey fieldKey : purgeList) {
                fiche.setSection(fieldKey, null);
            }
        }
    }

}
